<?php
/**
 * Main class
 *
 * @author Yithemes
 * @package YITH WooCommerce Customize My Account Page
 * @version 1.0.0
 */


if ( ! defined( 'YITH_WCMAP' ) ) {
	exit;
} // Exit if accessed directly

if ( ! class_exists( 'YITH_WCMAP' ) ) {
	/**
	 * YITH WooCommerce Customize My Account Page
	 *
	 * @since 1.0.0
	 */
	class YITH_WCMAP {

		/**
		 * Single instance of the class
		 *
		 * @var \YITH_WCMAP
		 * @since 1.0.0
		 */
		protected static $instance;

		/**
		 * Plugin version
		 *
		 * @var string
		 * @since 1.0.0
		 */
		public $version = YITH_WCMAP_VERSION;


		/**
		 * Returns single instance of the class
		 *
		 * @return \YITH_WCMAP
		 * @since 1.0.0
		 */
		public static function get_instance(){
			if( is_null( self::$instance ) ){
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * @return mixed YITH_WCMAP_Admin | YITH_WCMAP_Frontend
		 * @since 1.0.0
		 */
		public function __construct() {

			// Load Plugin Framework
			add_action( 'after_setup_theme', array( $this, 'plugin_fw_loader' ), 1 );
			// add endpoints
			add_action( 'after_setup_theme', array( $this, 'add_custom_endpoints' ), 99 );

			// rewrite rules
			add_action( 'init', array( $this, 'rewrite_rules' ), 20 );

			// Class admin
			if ( is_admin() && ! ( defined( 'DOING_AJAX' ) && DOING_AJAX && isset( $_REQUEST['context'] ) && $_REQUEST['context'] == 'frontend' ) ) {
				YITH_WCMAP_Admin();
			}
			// Class frontend
			else {
				YITH_WCMAP_Frontend();
			}

			// filter user avatar
			add_filter( 'get_avatar', array( $this, 'get_avatar' ), 10, 6 );

			// add endpoints to default
			add_filter( 'yith_wcmap_get_default_endpoints_keys_array', array( $this, 'endpoints_keys_array' ), 10, 1 );
			add_filter( 'yith_wcmap_get_default_endpoints_array', array( $this, 'endpoints_array' ), 10, 1 );

		}

		/**
		 * Load Plugin Framework
		 *
		 * @since  1.0
		 * @access public
		 * @return void
		 * @author Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function plugin_fw_loader() {

			if ( ! defined( 'YIT_CORE_PLUGIN' ) ) {
				global $plugin_fw_data;
				if( ! empty( $plugin_fw_data ) ){
					$plugin_fw_file = array_shift( $plugin_fw_data );
					require_once( $plugin_fw_file );
				}
			}
		}

		/**
		 * Add custom endpoints to main WC array
		 *
		 * @access public
		 * @since 1.0.0
		 * @author Francesco Licandro
		 */
		public function add_custom_endpoints(){

			$keys = yith_wcmap_get_endpoints_slug();

			if( empty( $keys ) || ! is_array( $keys ) ) {
				return;
			}

			foreach( $keys as $key ) {
				if( $key == 'dashboard' && isset( WC()->query->query_vars[$key] ) ){
					continue;
				}

				WC()->query->query_vars[$key] = $key;
			}


		}

		/**
		 * Rewrite rules
		 *
		 * @access public
		 * @since 1.0.0
		 * @author Francesco Licandro
		 */
		public function rewrite_rules(){

			$do_flush = get_option( 'yith-wcmap-flush-rewrite-rules', 1 );

			if( $do_flush ) {
				// change option
				update_option( 'yith-wcmap-flush-rewrite-rules', 0 );
				// the flush rewrite rules
				flush_rewrite_rules();
			}

		}

		/**
		 * Get customer avatar for user
		 *
		 * @access public
		 * @since 1.0.0
		 * @author Francesco Licandro
		 */
		public function get_avatar( $avatar, $user, $size, $default, $alt, $args ) {

			if( is_email( $user ) ) {
				$user = get_user_by( 'email', $user );
				$user = $user->ID;
			}

			// get custom avatar
			$custom_avatar = get_user_meta( $user, 'yith-wcmap-avatar', true );

			if( ! $custom_avatar ){
				return $avatar;
			}

			// maybe resize img
			$resized = yith_wcmap_resize_avatar_url( $custom_avatar, $size );
			// if error occurred return
			if( ! $resized ) {
				return $avatar;
			}

			$src = yith_wcmap_generate_avatar_url( $custom_avatar, $size );
			$class = array( 'avatar', 'avatar-' . (int) $args['size'], 'photo' );

			$avatar = sprintf(
				"<img alt='%s' src='%s' class='%s' height='%d' width='%d' %s/>",
				esc_attr( $args['alt'] ),
				esc_url( $src ),
				esc_attr( join( ' ', $class ) ),
				(int) $args['height'],
				(int) $args['width'],
				$args['extra_attr']
			);

			return $avatar;
		}

		/**
		 * Add endpoints keys ( like YITH Wishlist ecc.. ) to defaults
		 *
		 * @access public
		 * @since 1.0.0
		 * @param array $endpoints_keys
		 * @return array
		 * @author Francesco Licandro
		 */
		public function endpoints_keys_array( $endpoints_keys ) {

			if( defined( 'YITH_WCWL' ) && YITH_WCWL ) {
				$endpoints_keys[] = 'my-wishlist';
			}

			return $endpoints_keys;
		}

		/**
		 * Add custom plugin endpoints ( like YITH Wishlist ecc.. )
		 *
		 * @access public
		 * @since 1.0.0
		 * @param array $endpoints
		 * @return array
		 * @author Francesco Licandro
		 */
		public function endpoints_array( $endpoints ) {

			$plugins_endpoints = yith_wcmap_get_plugins_endpoints();

			return array_merge( $endpoints, $plugins_endpoints );
		}
	}
}

/**
 * Unique access to instance of YITH_WCMAP class
 *
 * @return \YITH_WCMAP
 * @since 1.0.0
 */
function YITH_WCMAP(){
	return YITH_WCMAP::get_instance();
}