<?php
/*
Plugin Name: WordPress Jobbin (wplist.org)
Plugin URI: http://getbutterfly.com/wordpress-plugins/jobbin/
Description: Offer job listings on your site with Jobbin! Your users can add jobs or browse jobs apply via the suggested method. The administrator can approve/deny any job and controls publishing status. The plugin offers manual payment for several types of jobs: full time, part time, freelance and internship/apprenticeship.
Author: Ciprian Popescu
Version: 0.2.5
Author URI: http://getbutterfly.com/
*/

error_reporting(0);

// Global constants
define('JOBBIN_VERSION', '0.2.5');
define('JOBBIN_JOB', 'jobbin_job');
define('JOBBIN_CATEGORY', 'jobbin_category');
define('JOBBIN_TITLE_SEPARATOR', ' &raquo; ');
define('JOBBIN_ADMIN_JOB_ENTRIES', 20);
define('JOBBIN_SEND_EMAIL', true);

//
if(!defined('WP_CONTENT_URL'))
	define('WP_CONTENT_URL', get_option('siteurl').'/wp-content');
if(!defined('WP_PLUGIN_URL'))
	define('WP_PLUGIN_URL', WP_CONTENT_URL.'/plugins');
if(!defined('WP_CONTENT_DIR'))
	define('WP_CONTENT_DIR', ABSPATH.'wp-content');
if(!defined('WP_PLUGIN_DIR'))
	define('WP_PLUGIN_DIR', WP_CONTENT_DIR.'/plugins');

//define('JOBBIN_PLUGIN_URL', WP_PLUGIN_URL.'/wp-jobbin');
//define('JOBBIN_PLUGIN_PATH', WP_PLUGIN_DIR.'/wp-jobbin');

$x = plugin_basename(__FILE__);
$x = WP_PLUGIN_URL.'/'.str_replace(basename( __FILE__),"",plugin_basename(__FILE__));

define('JOBBIN_PLUGIN_URL', $x);
define('JOBBIN_PLUGIN_PATH', $x);
//

// Includes
include('jobbin-template.php');

// Initialization functions
function jobbin_init() {
	load_plugin_textdomain('jobbin', false, JOBBIN_PLUGIN_PATH);
}
function add_jobbin_css() {
	wp_register_style('jobbinStyle', JOBBIN_PLUGIN_URL.'css/style.css');
	wp_enqueue_style('jobbinStyle');
}
function add_admin_css() {
	echo '<link type="text/css" rel="stylesheet" href="'.JOBBIN_PLUGIN_URL.'css/admin.css" />'."\n";
}

add_action('wp_print_styles', 'add_jobbin_css');
add_action('admin_head', 'add_admin_css');
add_action('init', 'jobbin_init');

function jobbin_preview($job) {
	ob_start();
	?>
	<form method="post" action="<?php echo jobbin_get_permalink();?>">
		<input type="hidden" name="jobbin_title" value="<?php echo esc_attr($job['title']);?>" />
		<input type="hidden" name="jobbin_category" value="<?php echo esc_attr($job['category']);?>" />
		<input type="hidden" name="jobbin_type" value="<?php echo esc_attr($job['type']);?>" />
		<input type="hidden" name="jobbin_description" value="<?php echo esc_attr($job['description']);?>" />
		<input type="hidden" name="jobbin_how_to_apply" value="<?php echo esc_attr($job['how_to_apply']);?>" />
		<input type="hidden" name="jobbin_location" value="<?php echo esc_attr($job['location']);?>" />
		<input type="hidden" name="jobbin_company_name" value="<?php echo esc_attr($job['company_name']);?>" />
		<input type="hidden" name="jobbin_company_url" value="<?php echo esc_attr($job['company_url']);?>" />
		<input type="hidden" name="jobbin_contact_name" value="<?php echo esc_attr($job['contact_name']);?>" />
		<input type="hidden" name="jobbin_contact_email" value="<?php echo esc_attr($job['contact_email']);?>" />
		<input type="hidden" name="jobbin-action" value="postjob" />
		<p><?php _e('Are you sure you want to post this job?');?></p>
		<input type="submit" value="<?php echo esc_attr(__('Yes'));?>"> <a href="javascript:history.go(-1);"><?php _e('No');?></a>
    </form>

		<p>
			<?php
			$jobbin_options = get_option('jobbin_options');
			$price = intval($jobbin_options[esc_attr($job['type']).'_price']);
			if($price != '0') {
				$jID = date('YmdHis');
				echo '<p>The price for this job is <strong>'.$jobbin_options['currency'].''.$price.'</strong>.</p>';
				echo '<p><small>You will receive an email with further instructions on your job listing payment.</small></p>';
				//echo jobbin_paypal_form_quick($jID, esc_attr($job['title']), $price);
			}
			?>
		</p>
	<?php
	$content = ob_get_contents();
	ob_end_clean();
	return $content;
}

function jobbin_extract_fields($data) {
	$extract = array();
	$fields = array('title','category', 'type','description','how_to_apply','location','company_name','company_url','contact_name','contact_email');
	foreach($fields as $field) {
		$extract[$field] = trim($data['jobbin_' . $field]);
	}
	return $extract;
}

function jobbin_page_handler() {
	global $wpdb;
	global $wp, $wp_query, $wp_the_query;
	global $post;

	if(!$post) $post =& get_post($id = 0);
	if(!$post) return;

	$jobbin_options = get_option('jobbin_options');

	if(!isset($post->ID) or (intval($post->ID) != $jobbin_options['post_id'])) return;

	if($_SERVER['REQUEST_METHOD'] == 'POST') {
		$_POST = stripslashes_deep($_POST);
		if(isset($_POST['jobbin-action'])) {
			if($_POST['jobbin-action'] == 'postjob' and $jobbin_options['enable_frontend'] and $jobbin_options['enable_guests']) {
				$job = jobbin_extract_fields($_POST);
				// set defaults
				$job['ad_duration'] 	= $jobbin_options['duration'];
				$job['ad_currency'] 	= $jobbin_options['currency'];
				$job['ad_price'] 		= $jobbin_options[$job['type'].'_price'];
				$job['ad_paid'] 		= $job['ad_price'] == '0' ? 1 : 0;
				$job['ad_approved'] 	= 0;
				$job['ad_published'] 	= 1;
				if(jobbin_insert_job($job)) {
					if($job['ad_paid'] == 1)
						wp_redirect(jobbin_get_permalink('action=waiting-approval'));
					else
						wp_redirect(jobbin_get_permalink('action=waiting-approval&pay=1'));
				}
				else {
					$description = apply_filters('the_content', $jobbin_options['description']);
					remove_filter('the_content', 'wptexturize');
					remove_filter('the_content', 'wpautop');

					$post->post_title = $jobbin_options['title'];

					if(!$jobbin_options['enable_frontend']) {
						$post->post_content = __('Job posting is disabled.', 'jobbin');
					}
					else {
						$error = __('<p>An error occured while posting your job! <strong>All fields are mandatory except where marked as optional.</strong> Please check your submission and try again. If the problem persists try contacting the administrator.</p>', 'jobbin');
						$post->post_content = $error . $description . jobbin_postjob_form(jobbin_extract_fields($_POST));
					}

					add_filter('wp_title', create_function('$title', "return \$title;"), 10);
				}
				return;
			}

			if($_POST['jobbin-action'] == 'preview') {
				$preview = jobbin_extract_fields($_POST);
				if(jobbin_valid_job($preview)) {
					$preview['title'] = sprintf(__('Preview: %s'), $preview['title']);
					$post->post_title = $preview['title'];
					$post->post_content = apply_filters('the_content', jobbin_view_job($preview)).jobbin_preview(jobbin_extract_fields($_POST));
				}
				else {
					$post->post_title = $jobbin_options['title'];
					if(!$jobbin_options['enable_frontend']) {
						$post->post_content = __('Job posting is disabled.', 'jobbin');
					}
					else {
						$error = __('<p>An error occured while posting your job! <strong>All fields are mandatory except where marked as optional.</strong> Please check your submission and try again. If the problem persists try contacting the administrator.</p>', 'jobbin');
						$post->post_content = $error . apply_filters('the_content', $jobbin_options['description']) . jobbin_postjob_form(jobbin_extract_fields($_POST));
					}
				}
				remove_filter('the_content', 'wptexturize');
				remove_filter('the_content', 'wpautop');
				add_filter('wp_title', create_function('$title', "return \$title;"), 10);
				return;
			}
		}

		if(isset($_POST['txn_id'])) {
			jobbin_paypal_ipn();
			return;
		}
	}

	if(!defined('JOBBIN_TITLE_SEPARATOR')) define('JOBBIN_TITLE_SEPARATOR', ' &raquo; ');

	if(isset($_GET['job_id']) and !isset($_GET['action'])) {
		if(jobbin_job_is_viewable(intval($_GET['job_id']))) {
			$job = jobbin_get_job(intval($_GET['job_id']));
			$post->post_title = $job['title'];
			$post->post_content = jobbin_view_job($job);
		}
		else {
			$post->post_title = __('Job not found', 'jobbin');
			$post->post_content = __('<p>The job you are looking for does not exist.</p>', 'jobbin');
		}
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		return;
	}

	if(isset($_GET['action']) and $_GET['action'] == 'postjob' and $jobbin_options['enable_guests']) {
		$description = apply_filters('the_content', $jobbin_options['description']);
		remove_filter('the_content', 'wptexturize');
		remove_filter('the_content', 'wpautop');

		$job['category'] = (isset($_GET['cat_ID'])) ? $_GET['cat_ID'] : $jobbin_options['default_category'];
		$post->post_title = $jobbin_options['title'];

		if(!$jobbin_options['enable_frontend']) {
			$post->post_content = __('Job posting is disabled.', 'jobbin');
		}
		else {
			$post->post_content = $description.jobbin_postjob_form($job);
		}

		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		return;
	}

	if(isset($_GET['action']) and $_GET['action'] == 'waiting-approval') {
		$post->post_title = __('Your job ad is waiting for approval.', 'jobbin');
		if($_GET['pay'] == 1)
			$post->post_content = '<p>'.__('Your job is pending reviewal. You will receive an email with payment instructions once your job has been approved. Thank you for your patience.', 'jobbin').'</p>';
		else
			$post->post_content = '<p>'.__('Your job is pending reviewal. Thank you for your patience.', 'jobbin').'</p>';
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		return;
	}

	if(isset($_GET['action']) and $_GET['action'] == 'paypal' and isset($_GET['job_id'])) {
		if(jobbin_job_paid(intval($_GET['job_id']))) {
			$post->post_title = __('Pay with PayPal', 'jobbin');
			$post->post_content = '<p>'.__('This job has already been paid.', 'jobbin').'</p>';
		}
		else {
			$job = jobbin_get_job(intval($_GET['job_id']));
			$post->post_title = __('Pay with PayPal', 'jobbin');
			$post->post_content = '<p>'.$jobbin_options['terms'].'</p>'.jobbin_paypal_form($job);
		}
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		return;
	}

	if(isset($_GET['action']) and $_GET['action'] == 'paypal-return') {
		$post->post_title = __('Your payment is being processed.', 'jobbin');
		$post->post_content = jobbin_paypal_return();
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		return;
	}

	if(isset($_GET['search'])) {
		$post->post_title = $jobbin_options['title'];
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		$post->post_content = jobbin_job_search($_GET['search']);
		return;
	}

	if(isset($_GET['jobfeed'])) {
		$jobfeed = $_GET['jobfeed'];
		if($jobfeed == 'rss2') {
			$cat_ID = isset($_GET['cat_ID']) ? intval($_GET['cat_ID']) : null;
			jobbin_rss2($cat_ID);
			exit;
		}
	}

	if(isset($_GET['cat'])) {
		$post->post_title = $jobbin_options['title'];
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		$post->post_content = jobbin_job_listing(mysql_real_escape_string($_GET['cat']));
		return;
	}
	if(!isset($_GET['cat'])) {
		$post->post_title = $jobbin_options['title'];
		add_filter('wp_title', create_function('$title', "return \$title;"), 10);
		$post->post_content = jobbin_job_listing();
		return;
	}
}

add_action('wp', 'jobbin_page_handler');

function jobbin_random_ad($content) {
	$jobbin_options = get_option('jobbin_options');

	if(!is_single() or !$jobbin_options['enable_random_ad']) return $content;

	global $wpdb;

	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;

	$columns = "$table_job.id, $table_job.posted, $table_job.title, $table_job.company_name, $table_job.company_url, $table_category.id AS category_id, $table_category.name AS category_name";
	$now = gmdate('Y-m-d H:i:s', time());

	$sql = "SELECT id FROM $table_job WHERE ad_approved = 1 AND ad_paid = 1 AND ad_published = 1 AND expired > '$now'";
	$job_IDs = $wpdb->get_col($sql);

	if(is_array($job_IDs) and count($job_IDs)) {
		$job_ID = $job_IDs[rand(0, count($job_IDs) - 1)];
		$sql = "SELECT $columns FROM $table_job INNER JOIN $table_category ON $table_job.category = $table_category.id WHERE $table_job.id = $job_ID";
		$job = $wpdb->get_row($sql);
	}

	if($job) {
		ob_start();
		jobbin_template_random_ad(array(
			'job' => $job
		));
		$random_ad = ob_get_contents();
		ob_end_clean();
		return $content.$random_ad;
	}
	else {
		return $content;
	}
}

add_action('the_content', 'jobbin_random_ad');

function jobbin_get_edit_post_link($link) {
	global $post;
	if(!isset($post)) $post =& get_post($id = 0);
	$jobbin_options = get_option('jobbin_options');
	if(isset($post->ID) and $post->ID == $jobbin_options['post_id']) {
		if(isset($_GET['job_id']))
			return get_bloginfo('wpurl').'/wp-admin/admin.php?page=jobbin-admin-jobs&amp;action=edit&amp;job_ID='.$_GET['job_id'];
		else
			return $link;
	}
	else {
		return $link;
	}
}
add_filter('get_edit_post_link', 'jobbin_get_edit_post_link', 1);

// Front-end functions
function jobbin_get_permalink($params = '') {
	global $wp_rewrite;

	$jobbin_options = get_option('jobbin_options');
	$post_id = $jobbin_options['post_id'];

	if(empty($wp_rewrite->permalink_structure)) {
		if($params)
			return get_permalink($post_id).'&'.$params;
		return get_permalink($post_id);
	}
	if($params)
		return get_permalink($post_id).'?'.$params;
	return get_permalink($post_id);
}

function jobbin_selected($current, $option) {
	if($current == $option) echo 'selected="selected"';
}

function jobbin_checked($current, $checked) {
	if($current == $checked) echo 'checked="checked"';
}

function jobbin_view_job($job) {
	ob_start();
	jobbin_template_view_job(array(
		'job' => $job
	));
	$content = ob_get_contents();
	ob_end_clean();
	return $content;
}

function jobbin_postjob_form($job = null) {
	$jobbin_options = get_option('jobbin_options');
	$categories = jobbin_get_all_categories();
	$types = jobbin_get_all_types();
	ob_start();
	?>
	<form method="post" action="<?php echo jobbin_get_permalink();?>">
		<h3><?php _e('Job Details', 'jobbin');?></h3>

		<label for="jobbin_title"><?php _e('Job title', 'jobbin');?></label><br />
		<input type="text" id="jobbin_title" name="jobbin_title" size="40" /><br />

		<label for="jobbin_category"><?php _e('Job category', 'jobbin');?></label><br />
		<select id="jobbin_category" name="jobbin_category">
			<?php foreach($categories as $cat) :?>
				<option value="<?php echo esc_attr($cat['id']) ?>"><?php echo esc_attr($cat['name']);?></option>
			<?php endforeach;?>
		</select><br />

		<label for="jobbin_type"><?php _e('Job type', 'jobbin');?></label><br />
		<select id="jobbin_type" name="jobbin_type">
			<?php foreach($types as $type) :?>
				<option value="<?php echo esc_attr($type['id'])?>"><?php echo esc_attr($type['name']);?></option>
			<?php endforeach;?>
		</select><br />

		<label for="jobbin_description"><?php _e('Description', 'jobbin');?></label><br />
		<textarea id="jobbin_description" name="jobbin_description" rows="8" cols="50"></textarea><br />

		<label for="jobbin_how_to_apply"><?php _e('How to apply', 'jobbin');?></label><br />
		<textarea id="jobbin_how_to_apply" name="jobbin_how_to_apply" rows="4" cols="50"></textarea><br />

		<label for="jobbin_location"><?php _e('Job location', 'jobbin');?></label><br />
		<input type="text" id="jobbin_location" name="jobbin_location" size="40" /><br />
		<br />
		<h3><?php _e('Contact Details', 'jobbin');?></h3>

		<label for="jobbin_company_name"><?php _e('Company name', 'jobbin');?></label><br />
		<input type="text" id="jobbin_company_name" name="jobbin_company_name" size="40" /><br />

		<label for="jobbin_company_url"><?php _e('Company URL (optional)', 'jobbin');?></label><br />
		<input type="text" id="jobbin_company_url" name="jobbin_company_url" size="40" /><br />

		<label for="jobbin_contact_name"><?php _e('Contact name', 'jobbin');?></label><br/>
		<input type="text" id="jobbin_contact_name" name="jobbin_contact_name" size="40" /><br />

		<label for="jobbin_contact_email"><?php _e('Contact/application email', 'jobbin');?></label><br />
		<input type="text" id="jobbin_contact_email" name="jobbin_contact_email" size="40" /><br />
		<br />
		<h3><?php _e('Terms and Conditions', 'jobbin');?></h3>
		<?php echo wpautop(wptexturize($jobbin_options['terms']));?>
		<input type="hidden" name="jobbin-action" value="preview" />
		<p><input id="submit" type="submit" value="<?php echo esc_attr(__('Add job', 'jobbin'));?>" /></p>
	</form>
	<?php
	$content = ob_get_contents();
	ob_end_clean();
	return $content;
}

function jobbin_job_listing($cat = null) {
	global $wpdb;

	$jobbin_options = get_option('jobbin_options');

	$jobs = jobbin_get_all_jobs($cat);

	$date_format = $jobbin_options['date_format'];
	$gmt_offset = intval(get_option('gmt_offset')) * 3600;

	$invite = empty($jobbin_options['invite']) ? esc_attr(_('Post your job now!')) : esc_attr($jobbin_options['invite']);
	$widget_invite = empty($jobbin_options['widget_invite']) ? esc_attr(__('Add a job', 'jobbin')) : esc_attr($jobbin_options['widget_invite']);

	ob_start();
	jobbin_template_job_listing(array(
		'enable_frontend' => $jobbin_options['enable_frontend'],
		'enable_guests' => $jobbin_options['enable_guests'],
		'jobs' => $jobs,
		'cat_ID' => $cat,
		'invite' => $invite,
		'widget_invite' => $widget_invite,
		'date_format' => $date_format,
		'gmt_offset' => $gmt_offset
	));
	$content = ob_get_contents();
	ob_end_clean();
	return $content;
}

function jobbin_job_search($query) {
	global $wpdb;

	$jobbin_options = get_option('jobbin_options');

	$jobs = jobbin_search_all_jobs($query);

	$date_format = $jobbin_options['date_format'];
	$gmt_offset = intval(get_option('gmt_offset')) * 3600;

	$invite = empty($jobbin_options['invite']) ? esc_attr(_('Post your job now!')) : esc_attr($jobbin_options['invite']);
	$widget_invite = empty($jobbin_options['widget_invite']) ? esc_attr(__('Add a job', 'jobbin')) : esc_attr($jobbin_options['widget_invite']);

	ob_start();
	jobbin_template_job_listing(array(
		'enable_frontend' => $jobbin_options['enable_frontend'],
		'enable_guests' => $jobbin_options['enable_guests'],
		'jobs' => $jobs,
		'cat_ID' => $cat,
		'invite' => $invite,
		'widget_invite' => $widget_invite,
		'date_format' => $date_format,
		'gmt_offset' => $gmt_offset
	));
	$content = ob_get_contents();
	ob_end_clean();
	return $content;
}

function jobbin_rss2($cat_ID = null) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$columns = "$table_job.id, $table_job.posted, $table_job.title, $table_job.description, $table_job.how_to_apply, $table_job.company_name, $table_job.location, $table_job.company_url, $table_category.id AS category_id, $table_category.name AS category_name, $table_job.type";
	$now = gmdate('Y-m-d H:i:s', time());
	if($cat_ID) {
		$cat_ID = intval($cat_ID);
		$jobs_sql = "SELECT $columns FROM $table_job INNER JOIN $table_category ON $table_job.category = $table_category.id WHERE ad_approved = 1 AND ad_paid = 1 AND ad_published = 1 AND (expired > '$now' OR ad_duration = -1) AND $table_category.id = $cat_ID ORDER BY posted DESC";
		$jobs = $wpdb->get_results($jobs_sql, ARRAY_A);
		$posted_sql = "SELECT MAX($table_job.posted) AS posted FROM $table_job WHERE ad_approved = 1 AND ad_paid = 1 AND ad_published = 1 AND (expired > '$now' OR ad_duration = -1) AND $table_category.id = $cat_ID ORDER BY posted DESC";
		$posted = $wpdb->get_var($posted_sql);
	}
	else {
		$jobs_sql = "SELECT $columns FROM $table_job INNER JOIN $table_category ON $table_job.category = $table_category.id WHERE ad_approved = 1 AND ad_paid = 1 AND ad_published = 1 AND (expired > '$now' OR ad_duration = -1) ORDER BY posted DESC";
		$jobs = $wpdb->get_results($jobs_sql, ARRAY_A);
		$posted_sql = "SELECT MAX($table_job.posted) AS posted FROM $table_job WHERE ad_approved = 1 AND ad_paid = 1 AND ad_published = 1 AND (expired > '$now' OR ad_duration = -1) ORDER BY posted DESC";
		$posted = $wpdb->get_var($posted_sql);
	}

	$jobbin_options = get_option('jobbin_options');

	header('Content-Type: text/xml; charset="'.get_option('blog_charset').'"', true);
	?>
<?php echo '<?xml version="1.0" encoding="'.get_option('blog_charset').'"?'.'>';?>
<rss version="2.0"
	xmlns:content="http://purl.org/rss/1.0/modules/content/"
	xmlns:wfw="http://wellformedweb.org/CommentAPI/"
	xmlns:dc="http://purl.org/dc/elements/1.1/"
	xmlns:atom="http://www.w3.org/2005/Atom"
	xmlns:sy="http://purl.org/rss/1.0/modules/syndication/"
>
<channel>
	<title><?php echo esc_attr($jobbin_options['title']);?></title>
	<link><?php echo jobbin_get_permalink();?></link>
	<description><?php echo esc_attr($jobbin_options['description']);?></description>
	<pubDate><?php echo mysql2date('D, d M Y H:i:s +0000', $posted, false);?></pubDate>
	<language><?php echo get_option('rss_language');?></language>
	<sy:updatePeriod>hourly</sy:updatePeriod>
	<sy:updateFrequency>1</sy:updateFrequency>

	<?php foreach($jobs as $job) :?>
	<item>
		<title><?php echo esc_attr($job['title']);?></title>
		<link><?php echo esc_attr(jobbin_get_permalink('job_id='.$job['id']));?></link>
		<pubDate><?php echo mysql2date('D, d M Y H:i:s +0000', $job['posted'], false);?></pubDate>
		<category><![CDATA[<?php echo html_entity_decode($job['category_name']);?>]]></category>
		<guid isPermaLink="false"><?php echo esc_attr(jobbin_get_permalink('job_id='.$job['id']));?></guid>
		<content:encoded><![CDATA[<?php echo $job['description'];?>]]></content:encoded>
	</item>
	<?php endforeach;?>
</channel>
</rss>
<?php
}

// HTTP & Net functions
function jobbin_http_request($method, $url, $data = '', $headers = array(), $timeout = 5) {
	$url = parse_url($url);
	if(!$url['path']) $url['path'] = '/';
	if($url['query']) $url['path'] .= '?'.$url['query'];
	$request = strtoupper($method).' '.$url['path']." HTTP/1.0\r\n";
	$headers['Host'] = $url['host'];
	$headers['Content-Length'] = strlen($data);
	foreach ($headers as $name => $value) {
		$request .= $name.': '.$value."\r\n";
	}
	$request .= "\r\n";
	$request .= $data;
	$response = false;
	if(!isset($url['port'])) $url['port'] = 80;
	if(false != ($http = @fsockopen($url['host'], $url['port'], $errno, $errstr, $timeout)) && is_resource($http)) {
		fwrite($http, $request);
		while(!feof($http))
			$response .= fgets($http, 1160); // One TCPIP packet
		fclose($http);
		$response = explode("\r\n\r\n", $response, 2);
	}
	return $response;
}

function jobbin_http_get($url, $data = '', $headers = array(), $timeout = 5) {
	if($data) $url .= '?'.$data;
	return jobbin_http_request('GET', $url, '', $headers, $timeout);
}

function jobbin_http_post($url, $data = '', $headers = array(), $timout = 5) {
	if(!isset($headers['Content-Type'])) {
		$headers = array_merge($headers, array('Content-Type' => 'application/x-www-form-urlencoded'));
	}
	return jobbin_http_request('POST', $url, $data, $headers, $timeout);
}

function jobbin_paypal_verify($data = '', $headers = array(), $timeout = 30) {
	$jobbin_options = get_option('jobbin_options');
	$url = parse_url($jobbin_options['paypal_verification_url']);
	if(!$url['path']) $url['path'] = '/';
	if($url['query']) $url['path'] .= '?'.$url['query'];
	$request = 'POST '.$url['path']." HTTP/1.0\r\n";
	$headers['Host'] = $url['host'];
	$headers['Content-Length'] = strlen($data);
	foreach($headers as $name => $value) {
		$request .= $name.': '.$value."\r\n";
	}
	$request .= "\r\n";
	$request .= $data;
	$response = false;
	if(!isset($url['port'])) $url['port'] = 443;
	if(false != ($http = fsockopen($url['scheme'].'://'.$url['host'], $url['port'], $errno, $errstr, $timeout)) && is_resource($http)) {
		fwrite($http, $request);
		while(!feof($http))
			$response .= fgets($http, 1160); // One TCPIP packet
		fclose($http);
		$response = explode("\r\n\r\n", $response, 2);
	}
	return $response;
}

// Administrative pages
function jobbin_install_page() {
	global $user_ID;

	$post['post_type'] 		= 'page';
	$post['post_title'] 	= __('Jobs', 'jobbin');
	$post['post_name'] 		= __('jobs', 'jobbin');
	$post['post_content'] 	= __('This page was auto-generated by Jobbin during installation. Please do not delete this page.', 'jobbin');
	$post['post_excerpt'] 	= '';
	$post['post_parent'] 	= 0;
	$post['to_ping'] 		= '';
	$post['post_author'] 	= $user_ID;
	$post['post_status'] 	= 'publish';
	$post['comment_status'] = 'closed';
	$post['ping_status'] 	= 'closed';

	$post_ID = wp_insert_post($post);
	if(is_wp_error($post_ID))
		return $post_ID;

	if(empty($post_ID))
		return 0;

	return $post_ID;
}

function jobbin_install() {
	global $wpdb;

	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;

	// THIS ADDS 'workingfromhome' TO DATABASE FROM 0.2.3
	// THE VERSION UPDATE SHOULD STAY IN PLACE FOR ALL UPDATES
	/*
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$jobbin_options = get_option('jobbin_options');
	if($jobbin_options['version'] != '0.2.4') {
		$sql = "ALTER TABLE `$table_job` CHANGE `type` `type` ENUM('fulltime','parttime','freelance','internship','apprenticeship','workingfromhome') NOT NULL DEFAULT 'fulltime'";
		$wpdb->query($sql);
		require_once(ABSPATH.'wp-admin/includes/upgrade.php');

		update_option('version', '0.2.4');
	}
	*/
	// END UPGRADE

	if($wpdb->get_var("SHOW TABLES LIKE '$table_category'") != $table_category) {
		$sql = "CREATE TABLE $table_category (
					id int(4) unsigned NOT NULL auto_increment,
					name varchar(255) NOT NULL,
					priority int(10) unsigned NOT NULL default '10',
					job_count bigint(20) unsigned NOT NULL default '0',
					PRIMARY KEY (id)
				);";

		require_once(ABSPATH.'wp-admin/includes/upgrade.php');
		dbDelta($sql);

		$sql = "INSERT INTO $table_category (name) VALUES('".$wpdb->escape(__('Uncategorized', 'jobbin'))."')";
		$result = $wpdb->query($sql);

		$sql = "INSERT INTO $table_category (name) VALUES('".$wpdb->escape(__('General jobs', 'jobbin'))."')";
		$result = $wpdb->query($sql);

		$sql = "INSERT INTO $table_category (name) VALUES('".$wpdb->escape(__('Programming jobs', 'jobbin'))."')";
		$result = $wpdb->query($sql);

		$sql = "INSERT INTO $table_category (name) VALUES('".$wpdb->escape(__('Other', 'jobbin'))."')";
		$result = $wpdb->query($sql);
	}

	$table_job = $wpdb->prefix.JOBBIN_JOB;

	if($wpdb->get_var("SHOW TABLES LIKE '$table_job'") != $table_job) {
		$sql = "CREATE TABLE $table_job (
					id bigint(20) unsigned NOT NULL auto_increment,
					posted datetime NOT NULL default '0000-00-00 00:00:00',
					modified datetime NOT NULL default '0000-00-00 00:00:00',
					expired datetime NOT NULL default '0000-00-00 00:00:00',
					title varchar(255) NOT NULL default '',
					category int(4) unsigned NOT NULL default '1',
					`type` ENUM('fulltime','parttime','freelance','internship','apprenticeship','workingfromhome') NOT NULL DEFAULT 'fulltime',
					description text NOT NULL,
					how_to_apply text NOT NULL,
					location varchar(100) NOT NULL default '',
					company_name varchar(100) NOT NULL default '',
					company_url varchar(100) NOT NULL default '',
					contact_name varchar(100) NOT NULL default '',
					contact_email varchar(100) NOT NULL default '',
					ad_duration int(4) NOT NULL default '0',
					ad_currency char(3) NOT NULL default 'USD',
					ad_price double NOT NULL default '0',
					ad_paid BOOLEAN NOT NULL default '0',
					ad_approved BOOLEAN NOT NULL default '0',
					ad_published BOOLEAN NOT NULL default '1',
					ip_address varchar(100) NOT NULL default '',
					txn_id varchar(17),
					PRIMARY KEY (id)
				);";
		require_once(ABSPATH.'wp-admin/includes/upgrade.php');
		dbDelta($sql);
	}

	$jobbin_options = get_option('jobbin_options');
	if(!empty($jobbin_options) and $jobbin_options['post_id']) {
		$post_ID = $jobbin_options['post_id'];
	}
	else {
		$post_ID = jobbin_install_page();
	}

	$default_options = array(
		'post_id' 					=> $post_ID,
		'version' 					=> JOBBIN_VERSION,
		'title' 					=> __('Job Board', 'jobbin'),
		'description' 				=> '',
		'invite' 					=> __('Post your job now!'),
		'enable_frontend' 			=> 1,
		'enable_guests' 			=> 1,
		'duration' 					=> 30,
		'currency' 					=> 'EUR',
		'fulltime_price' 			=> 0,
		'parttime_price' 			=> 0,
		'freelance_price' 			=> 0,
		'internship_price' 			=> 0,
		'apprenticeship_price' 		=> 0,
		'workingfromhome_price' 	=> 0,
		'paypal_email' 				=> '',
		'paypal_url' 				=> 'https://www.paypal.com/cgi-bin/webscr',
		'paypal_verification_url' 	=> 'ssl://www.paypal.com:443/cgi-bin/webscr',
		'date_format' 				=> 'n/j',
		'terms' 					=> '',
		'payment_email_subject' 	=> 'Job Board Notification',
		'payment_email_message' 	=> '',
		'publish_email_subject' 	=> 'Job Board Notification',
		'publish_email_message' 	=> '',
		'email_from_name' 			=> 'Jobbin',
		'email_from' 				=> 'jobbin@example.com',
		'viewable_expired_ads' 		=> 0,
		'enable_random_ad' 			=> 0,
		'default_category' 			=> 1,
		'widget_title' 				=> __('Job Board', 'jobbin'),
		'widget_invite' 			=> __('Add a job', 'jobbin')
	);

	foreach($default_options as $name => $value) {
		if(!isset($jobbin_options[$name])) {
			$jobbin_options[$name] = $value;
		}
	}

	$jobbin_options['version'] = JOBBIN_VERSION;
	update_option('jobbin_options', $jobbin_options);
}

register_activation_hook(__FILE__, 'jobbin_install');

function jobbin_uninstall() {
	global $wpdb;
	$jobbin_options = get_option('jobbin_options');
	$wpdb->query('DROP TABLE '.$wpdb->prefix.JOBBIN_CATEGORY);
	$wpdb->query('DROP TABLE '.$wpdb->prefix.JOBBIN_JOB);
	$wpdb->query('DELETE FROM '.$wpdb->prefix.'posts WHERE ID = '.$jobbin_options['post_id']);
	$wpdb->query('DELETE FROM '.$wpdb->prefix.'postmeta WHERE post_id = '.$jobbin_options['post_id']);
	delete_option('jobbin_options');
	deactivate_plugins('wp-jobbin/jobbin.php');
	wp_redirect('plugins.php?deactivate=true');
}
function jobbin_uninstall_keep() {
	deactivate_plugins('wp-jobbin/jobbin.php');
	wp_redirect('plugins.php?deactivate=true');
}

function jobbin_add_admin_pages() {
	global $wpdb;
	global $plugin_page, $pagenow;

	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$unapproved = intval($wpdb->get_var("SELECT COUNT(id) AS unapproved FROM $table_job WHERE ad_approved = 0"));

	$submenu = array();

	add_menu_page(__('Jobbin for WordPress', 'jobbin'), __('Jobbin', 'jobbin'), 'manage_options', __FILE__, 'jobbin_admin_index', JOBBIN_PLUGIN_URL.'images/icon-16.png');
	$submenu['jobbin_admin_load_jobs'] = add_submenu_page(__FILE__, __('Jobbin Listings', 'jobbin'), __('Jobs', 'jobbin'), 'manage_options', 'jobbin-admin-jobs', 'jobbin_admin_jobs');
	if($unapproved or (isset($_GET['page']) == 'jobbin-admin-approvals' and isset($_GET['message'])))
		$submenu['jobbin_admin_load_approvals'] = add_submenu_page(__FILE__, __('Jobbin Approvals', 'jobbin'), sprintf(__('Pending Jobs (%d)', 'jobbin'), $unapproved), 'manage_options', 'jobbin-admin-approvals', 'jobbin_admin_approvals');
	$submenu['jobbin_admin_load_categories'] = add_submenu_page(__FILE__, __('Jobbin Categories', 'jobbin'), __('Categories', 'jobbin'), 'manage_options', 'jobbin-admin-categories', 'jobbin_admin_categories');
	$submenu['jobbin_admin_load_options'] = add_submenu_page(__FILE__, __('Jobbin Options', 'jobbin'), __('Options', 'jobbin'), 'manage_options', 'jobbin-admin-options', 'jobbin_admin_options');
	$submenu['jobbin_admin_load_uninstall'] = add_submenu_page(__FILE__, __('Jobbin Uninstall', 'jobbin'), __('Uninstall', 'jobbin'), 'manage_options', 'jobbin-admin-uninstall', 'jobbin_admin_uninstall');

	foreach($submenu as $handler => $page_hook) {
		if($page_hook == get_plugin_page_hook($plugin_page, $pagenow)) {
			add_action('load-'.$page_hook, $handler);
		}
	}
}

add_action('admin_menu', 'jobbin_add_admin_pages');

function jobbin_admin_load_approvals() {
	if(isset($_POST['action'])) {
		$action = $_POST['action'];
		if($action == 'approve') return jobbin_admin_approve_job();
	}
}

function jobbin_admin_approvals() {
	$jobbin_options = get_option('jobbin_options');

	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$unapproved = $wpdb->get_results("SELECT * FROM $table_job WHERE ad_approved = 0 ORDER BY posted ASC", ARRAY_A);
	$gmt_offset = intval(get_option('gmt_offset')) * 3600;

	$messages[1] = __('Job approved.', 'jobbin');
	$messages[2] = __('Job not approved.', 'jobbin');

	if(isset($_GET['message'])) :
		?>
		<div id="message" class="updated fade"><p><?php echo $messages[$_GET['message']];?></p></div>
		<?php
	endif;
	?>

	<div class="wrap">
		<div id="icon-plugins" class="icon32"></div>
		<h2><?php _e('Approval Queue', 'jobbin');?></h2>
		<?php if(empty($unapproved)) :?>
			<p><?php _e('No jobs found.', 'jobbin');?></p>
		<?php else :?>
			<ol class="commentlist">
				<?php foreach($unapproved as $job) :?>
				<li>
					<strong><?php echo $job['title'];?></strong> (<small><?php _e('Website:', 'jobbin');?> <a href="<?php echo $job['company_url'];?>"><?php echo $job['company_name'];?></a> | <?php _e('Email:', 'jobbin');?> <a href="mailto:<?php echo $job['contact_email'];?>"><?php echo $job['contact_email'];?></a></small>) - <?php _e('IP:', 'jobbin');?> <?php echo $job['ip_address'];?> (<small><?php echo date('M d, g:i A', strtotime($job['posted']) + $gmt_offset);?>) - [<a href="admin.php?page=jobbin-admin-jobs&amp;action=edit&amp;job_ID=<?php echo esc_attr($job['id']);?>">Edit</a> or <a href="<?php echo 'admin.php?page=jobbin-admin-jobs&amp;action=delete&amp;job_ID='.$job['id'], 'delete-job_'.$job['id']?>">Delete</a>]</small>
					<br /><br />
					<div class="jobcontent" id="jobcontent-<?php echo $job['id'];?>">
						<strong>Job description:</strong> <?php echo $job['description'];?><br />
						<strong>Address:</strong> <?php echo $job['location'];?><br/>
						<strong>How to apply?</strong> <?php echo $job['how_to_apply'];?>
					</div>
					<form method="post" action="admin.php?page=jobbin-admin-approvals&amp;action=approve">
						<input type="hidden" name="job_ID" value="<?php echo esc_attr($job['id']);?>" />
						<input type="hidden" name="action" value="approve" />
						<p><input type="submit" class="button-primary" value="<?php _e('Approve', 'jobbin');?>"></p>
					</form>
				</li>
				<?php endforeach;?>
			</ol>
		<?php endif;?>
	</div>
<?php
}

function jobbin_admin_approve_job() {
	$_POST = stripslashes_deep($_POST);
	$job_ID = intval($_POST['job_ID']);
	$job = jobbin_get_job($job_ID);
	if(jobbin_set_approved($job)) {
		if(intval($job['ad_price']) == 0) {
			jobbin_send_publish_email($job);
		}
		else {
			jobbin_send_payment_email($job);
		}
		wp_redirect('admin.php?page=jobbin-admin-approvals&message=1');
	}
	else {
		wp_redirect('admin.php?page=jobbin-admin-approvals&message=2');
	}
}

function jobbin_set_approved($job, $ad_approved = 1) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$now = time();
	$modified = $wpdb->escape(gmdate('Y-m-d H:i:s', $now));
	$updates = array();
	$updates[] = "modified = '$modified'";
	$ad_approved = intval($ad_approved) ? 1 : 0;
	$updates[] = "ad_approved = $ad_approved";
	$expired = gmdate('Y-m-d H:i:s', $now + intval($job['ad_duration']) * 86400);
	$expired = $wpdb->escape($expired);
	$updates[] = "expired = '$expired'";
	$updates = implode(', ', $updates);
	return $wpdb->query("UPDATE $table_job SET $updates WHERE id = ".$job['id']);
}

function jobbin_set_paid($job, $txn_id, $ad_paid = 1) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$now = time();
	$modified = $wpdb->escape(gmdate('Y-m-d H:i:s', $now));
	$updates = array();
	$updates[] = "modified = '$modified'";
	$ad_paid = intval($ad_paid) ? 1 : 0;
	$updates[] = "ad_paid = $ad_paid";
	$expired = gmdate('Y-m-d H:i:s', $now + intval($job['ad_duration']) * 86400);
	$expired = $wpdb->escape($expired);
	$updates[] = "expired = '$expired'";
	$txn_id = $wpdb->escape($txn_id);
	$updates[] = "txn_id = '$txn_id'";
	$updates = implode(', ', $updates);
	return $wpdb->query("UPDATE $table_job SET $updates WHERE id = ".$job['id']);
}

function jobbin_job_paid($job_ID) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$ad_paid = $wpdb->get_var("SELECT ad_paid FROM $table_job WHERE id = ".$job_ID, 0);
	return $ad_paid == 1 ? true : false;
}

function jobbin_admin_index() {
	global $wpdb;

	$jobbin_options = get_option('jobbin_options');
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$total = intval($wpdb->get_var("SELECT COUNT(id) AS total FROM $table_job WHERE ad_approved = 1", 0));
	$url = urlencode(get_bloginfo('url'));
	?>
	<div class="wrap">
		<div id="icon-plugins" class="icon32"></div>
		<h2><?php _e('Overview', 'jobbin');?></h2>
		<?php include('jobbin-sidebar.php');?>
        <p>Welcome to <strong>WordPress Jobbin</strong>!</p>
        <p><strong>WordPress Jobbin</strong> is a job publishing plugin for multiple job types and categories.</p>
		<h3>What Now?</h3>
		<ol>
			<li>The plugin created a new page called Jobs. Do not modify or delete that page, as it is the placeholder for jobs, categories and Jobbin forms.</li>
			<li>Check the Options panel, enable job posting, and modify each setting according to your site's theme.</li>
			<li>Read below.</li>
		</ol>
		<h3>Adding Jobs</h3>
		<p>Jobs can be added both by administrator via the back-end form and by users via the front-end form. Filling in all text fields and text areas will make sure your jobs get maximum visibility.</p>
		<p>Keep your title simple and keywords rich.</p>
		<p>From the <strong>Jobs Listings</strong> panel you can modify or delete your existing jobs or job offers.</p>
		<h3>Approving Jobs</h3>
		<p>Whenever a job is pending, a new menu option will appear in your Jobbin section, <strong>Pending Jobs (x)</strong>, where (x) is the number of new jobs.</p>
		<p>You can view or edit the job before approving it, or simply delete it if it does not fit your site's theme or is simply spammy.</p>
 		<h3>Adding Jobs Categories</h3>
		<p>You can add an unlimited number of categories and set their desired order. We recommend no more than 20 categories for a decent site.</p>
 		<h3>Settings and Options</h3>
		<p>Customize your plugin's behaviour the way you want it! Email messages, PayPal email, job duration, job prices, terms and conditions.</p>
		<p>Check the <a href="http://www.blogtycoon.net/wordpress-plugins/jobbin/" rel="external">official homepage</a> for feedback and support.</p>
   </div>
<?php
}

function jobbin_admin_load_categories() {
	if(isset($_POST['action'])) {
		$action = $_POST['action'];
		if($action == 'add') return jobbin_admin_add_category();
		if($action == 'update') return jobbin_admin_update_category();
	}
}

function jobbin_admin_categories() {
	$jobbin_options = get_option('jobbin_options');

	if(isset($_GET['action'])) {
		$action = mysql_real_escape_string($_GET['action']);
		if($action == 'delete')
			return jobbin_admin_delete_category();
		if($action == 'edit')
			return jobbin_admin_edit_category();
	}
	$categories = jobbin_get_all_categories();

	$messages[1] = __('New category added sucessfully!', 'jobbin');
	$messages[2] = __('Category deleted.', 'jobbin');
	$messages[3] = __('Category updated.', 'jobbin');
	$messages[4] = __('Category not added.', 'jobbin');
	$messages[5] = __('Category not updated.', 'jobbin');

	$form = '<form name="addcat" id="addcat" method="post" action="admin.php?page=jobbin-admin-categories&action=add">';
	$action = 'add';
	$heading = __('Add Job Category', 'jobbin');
	$submit_text = __('Add Job Category', 'jobbin');

	if(isset($_GET['message'])) :
		?>
		<div id="message" class="updated fade"><p><?php echo $messages[$_GET['message']];?></p></div>
		<?php
	endif;
	?>

	<div class="wrap">
		<div id="icon-plugins" class="icon32"></div>
		<h2><?php _e('Job Categories', 'jobbin');?></h2>
		<table class="widefat boxin">
			<thead>
				<tr>
					<th>ID</th>
					<th>Name</th>
					<th>Jobs</th>
					<th>Priority (Order)</th>
					<th>Action</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach($categories as $i => $cat) :?>
					<?php if($i % 2) :?>
						<tr>
					<?php else :?>
						<tr class="alternate">
					<?php endif;?>
					<td><?php echo esc_attr($cat['id']);?></td>
					<td><?php echo esc_attr($cat['name']);?></td>
					<td><?php echo esc_attr($cat['job_count']);?></td>
					<td><?php echo esc_attr($cat['priority']);?></td>
					<td>
						<a href="admin.php?page=jobbin-admin-categories&amp;action=edit&amp;cat_ID=<?php echo $cat['id'];?>"><?php _e('Edit', 'jobbin');?></a>
						<?php if($cat['id'] != intval($jobbin_options['default_category'])) :?>
							 | <a href="admin.php?page=jobbin-admin-categories&amp;action=delete&amp;cat_ID=<?php echo $cat['id'];?>"><?php _e('Delete', 'jobbin');?></a>
						<?php endif;?>
					</td>
				</tr>
				<?php endforeach?>
			</tbody>
		</table>

		<p><?php printf(__('<strong>NOTE:</strong> Deleting a category does not delete the jobs in that category. Instead, jobs that were assigned to the deleted category are set to the category <strong>%s</strong>.', 'jobbin'), jobbin_get_catname($jobbin_options['default_category']));?></p>

		<h2><?php echo $heading;?></h2>
		<?php jobbin_admin_category_form(NULL, $form, $action, $submit_text);?>
	</div>
	<?php
}

function jobbin_admin_category_form($category, $form, $action, $submit_text) {
	global $wp_version;
	if(is_null($category)) {
		$category = array();
		$category['id'] = '';
		$category['name'] = '';
		$category['priority'] = 1;
	}
	?>
	<?php echo $form;?>
		<input type="hidden" value="<?php echo $action;?>" name="action" />
		<input type="hidden" value="<?php echo esc_attr($category['id']);?>" name="cat_ID" />
		<table class="widefat boxin">
			<tr>
				<th scope="row" valign="top"><label for="name"><?php _e('Category name:', 'jobbin');?></label></th>
				<td><input type="text" class="regular-text" id="name" name="name" size="40" value="<?php echo esc_attr($category['name']);?>" /></td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="priority"><?php _e('Priority:', 'jobbin');?></label></th>
				<td>
					<input type="text" id="priority" name="priority" size="3" value="<?php echo esc_attr($category['priority']) ?>" /> <span class="description"><?php _e('Determines the order of display.', 'jobbin');?></span>
				</td>
			</tr>
		</table>
		<p><input type="submit" name="Submit" class="button-primary" value="<?php echo esc_attr($submit_text);?>" /></p>
	</form>
	<?php
}

function jobbin_admin_load_jobs() {
	if(isset($_GET['action'])) {
		$action = $_GET['action'];
		if($action == 'add') return jobbin_admin_add_job();
		if($action == 'update') return jobbin_admin_update_job();
		if($action == 'delete') return jobbin_admin_delete_job();
	}
}

function jobbin_admin_jobs() {
	$jobbin_options = get_option('jobbin_options');

	if(isset($_GET['action'])) {
		$action = $_GET['action'];
		if($action == 'edit')
			return jobbin_admin_edit_job();
	}

	global $wpdb, $wp_version;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$columns = 'id, posted, modified, title, category, type, company_name, company_url, ad_approved, ad_paid, ad_published, ad_currency, ad_price';

	$page = max(isset($_GET['paged']) ? intval($_GET['paged']) : 1, 1);
	$limit = JOBBIN_ADMIN_JOB_ENTRIES;
	$offset = ($page - 1) * $limit;

	if(isset($_GET['s'])) {
		$search = $wpdb->escape($_GET['s']);
		$sql = "SELECT $columns FROM $table_job WHERE title LIKE '%$search%' ORDER BY posted DESC LIMIT $limit OFFSET $offset";
		$jobs = $wpdb->get_results($sql, ARRAY_A);
		$total = intval($wpdb->get_var("SELECT COUNT(id) AS total FROM $table_job WHERE title LIKE '%$search%' ORDER BY posted DESC", 0));
	}
	else {
		$sql = "SELECT $columns FROM $table_job ORDER BY posted DESC LIMIT $limit OFFSET $offset";
		$jobs = $wpdb->get_results($sql, ARRAY_A);
		$total = intval($wpdb->get_var("SELECT COUNT(id) AS total FROM $table_job ORDER BY posted DESC", 0));
	}

	if(!$jobs) $jobs = array();

	$prev = $offset + $limit < $total ? true : false;
	$next = $page > 1 ? true : false;

	$messages[1] = __('Job added.', 'jobbin');
	$messages[2] = __('Job deleted.', 'jobbin');
	$messages[3] = __('Job updated.', 'jobbin');
	$messages[4] = __('Job not added.', 'jobbin');
	$messages[5] = __('Job not updated.', 'jobbin');

	$form = '<form name="addjob" id="addjob" method="post" action="admin.php?page=jobbin-admin-jobs&action=add">';
	$action = 'add';
	$heading = __('Add Job Listing', 'jobbin');
	$submit_text = __('Add Job Listing', 'jobbin');
	$gmt_offset = intval(get_option('gmt_offset')) * 3600;

	$colors = array('fulltime' => '#999933', 'parttime' => '#3399FF', 'freelance' => '#CC6600', 'internship' => '#660033', 'apprenticeship' => '#660033', 'workingfromhome' => '#CCCCCC');
	$labels = array('fulltime' => __('Full Time', 'jobbin'), 'parttime' => __('Part Time', 'jobbin'), 'freelance' => __('Freelance', 'jobbin'), 'internship' => __('Internship', 'jobbin'), 'apprenticeship' => __('Apprenticeship', 'jobbin'), 'workingfromhome' => __('Working from home', 'jobbin'));
	?>
	<?php if(isset($_GET['message'])) :?>
		<div id="message" class="updated fade"><p><?php echo $messages[$_GET['message']];?></p></div>
	<?php endif;?>

	<div class="wrap">
		<div id="icon-plugins" class="icon32"></div>
		<h2><?php _e('Job Listing <small>(<a href="#addjob">add new</a>)</small>', 'jobbin');?></h2>
		<form method="get">
			<ul class="subsubsub">
				<li><a class="current"><?php _e('All', 'jobbin');?></a></li>
			</ul>
			<p class="search-box">
				<input type="hidden" name="page" value="<?php echo esc_attr(stripslashes($_GET['page']));?>">
				<input type="text" class="search-input" name="s" value="" />
				<input type="submit" value="Search Jobs" class="button" />
			</p>
			<div class="clear" />
		</form>
		<br class="clear" />
		<table class="widefat boxin">
			<thead>
				<tr>
					<th><?php _e('ID', 'jobbin');?></th>
					<th><?php _e('When', 'jobbin');?></th>
					<th><?php _e('Type', 'jobbin');?></th>
					<th><?php _e('Title', 'jobbin');?></th>
					<th><?php _e('Published', 'jobbin');?></th>
					<th><?php _e('Approved', 'jobbin');?></th>
					<th><?php _e('Price', 'jobbin');?></th>
					<th><?php _e('Paid', 'jobbin');?></th>
					<th><?php _e('Action', 'jobbin');?></th>
				</tr>
			</thead>
			<tbody>
				<?php if(empty($jobs)) :?>
					<tr><td colspan="7"><?php _e('No jobs found.', 'jobbin');?></td></tr>
				<?php endif;?>
				<?php foreach($jobs as $i => $job) :?>
					<?php if($i % 2) :?>
					<tr>
					<?php else :?>
					<tr class="alternate">
					<?php endif;?>
						<td><?php echo esc_attr($job['id']);?></td>
						<td><small>
							<?php echo esc_attr(date('Y-m-d', strtotime($job['posted']) + $gmt_offset));?><br />
							<?php echo esc_attr(date('g:i:s a', strtotime($job['posted']) + $gmt_offset));?>
						</small></td>
						<td><div class="tag" style="background-color: <?php echo $colors[$job['type']];?>"><?php echo esc_attr($labels[$job['type']]);?></div></td>
						<td><?php echo sprintf(__('%1$s at %2$s', 'jobbin'), esc_attr($job['title']), '<a href="'.esc_attr($job['company_url']).'">'.esc_attr($job['company_name'])).'</a>';?></td>
						<td><?php $job['ad_published'] ? _e('Yes') : _e('No');?></td>
						<td><?php $job['ad_approved'] ? _e('Yes') : _e('No');?></td>
						<?php if($job['ad_price']) :?>
							<td><?php echo $job['ad_currency'];?> <?php echo $job['ad_price'];?></td>
							<td><?php $job['ad_paid'] ? _e('Yes') : _e('No');?></td>
						<?php else :?>
							<td colspan="2"><?php _e('Free listing', 'jobbin');?></td>
						<?php endif;?>
						<td>
							<a href="admin.php?page=jobbin-admin-jobs&amp;action=edit&amp;job_ID=<?php echo $job['id'];?>"><?php _e('Edit', 'jobbin');?></a> | 
							<a href="admin.php?page=jobbin-admin-jobs&amp;action=delete&amp;job_ID=<?php echo $job['id'];?>"><?php _e('Delete', 'jobbin');?></a>
						</td>
					</tr>
				<?php endforeach;?>
			</tbody>
		</table>
		<div class="navigation">
			<?php if($prev) :?>
				<div class="alignleft">
					<a href="admin.php?page=jobbin-admin-jobs&amp;paged=<?php echo $page + 1;?>"><?php _e('&laquo; Previous Entries', 'jobbin');?></a>
				</div>
			<?php endif;?>
			<?php if($next) :?>
				<div class="alignright">
					<a href="admin.php?page=jobbin-admin-jobs&amp;paged=<?php echo $page - 1;?>"><?php _e('Next Entries &raquo;', 'jobbin');?></a>
				</div>
			<?php endif;?>
		</div>

		<h2><?php echo $heading;?></h2>
		<?php jobbin_admin_add_job_form(null, $form, $action, $submit_text);?>
	</div>

	<?php
}

function jobbin_admin_job_form($job, $form, $action, $submit_text) {
	global $wp_version,$now;
	$categories = jobbin_get_all_categories();
	$types = jobbin_get_all_types();
	$currencies = jobbin_get_all_currencies();
	$gmt_offset = intval(get_option('gmt_offset')) * 3600;
	$date_format = get_option('date_format');
	$time_format = get_option('time_format');
	gmdate('Y-m-d H:i:s', $now + intval($job['ad_duration']) * 86400);
	?>
	<?php echo $form;?>
		<input type="hidden" value="<?php echo $action;?>" name="action" />
		<input type="hidden" value="<?php echo esc_attr($job['id']) ?>" name="job_ID" />
		<table class="form-table">
			<tr>
				<th scope="row" valign="middle"><label for="title"><?php _e('Job title:', 'jobbin');?></label></th>
				<td><input type="text" class="regular-text" id="title" name="title" size="40" value="<?php echo esc_attr($job['title']);?>" /></td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="category"><?php _e('Category:', 'jobbin');?></label></th>
				<td>
					<select id="category" name="category">
						<?php foreach($categories as $cat) :?>
							<option value="<?php echo esc_attr($cat['id']);?>" <?php jobbin_selected($job['category'], $cat['id']);?>><?php echo esc_attr($cat['name']);?></option>
						<?php endforeach;?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="type"><?php _e('Type:', 'jobbin');?></label></th>
				<td>
					<select id="type" name="type">
						<?php foreach($types as $type) :?>
							<option value="<?php echo esc_attr($type['id']);?>" <?php if(isset($job['type'])) jobbin_selected($job['type'], $type['id']);?>><?php echo esc_attr($type['name']);?></option>
						<?php endforeach;?>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="description"><?php _e('Description:', 'jobbin');?></label></th>
				<td>
					<textarea id="description" name="description" rows="7" cols="30"><?php echo esc_attr($job['description']);?></textarea>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="how_to_apply"><?php _e('How to apply:', 'jobbin');?></label></th>
				<td>
					<textarea id="how_to_apply" name="how_to_apply" rows="4" cols="30"><?php echo esc_attr($job['how_to_apply']);?></textarea>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="location"><?php _e('Job location:', 'jobbin');?></label></th>
				<td><input type="text" class="regular-text" id="location" name="location" size="40" value="<?php echo esc_attr($job['location']);?>" /></td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="company_name"><?php _e('Company name:', 'jobbin');?></label></th>
				<td><input type="text" class="regular-text" id="company_name" name="company_name" size="40" value="<?php echo esc_attr($job['company_name']);?>" /></td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="company_url"><?php _e('Company URL (optional):', 'jobbin');?></label></th>
				<td>
					<input type="text" class="regular-text" id="company_url" name="company_url" size="40" value="<?php echo esc_attr($job['company_url']);?>" />
					<br/><span class="description"><?php _e("Don't forget the <strong>http://</strong> part.", 'jobbin');?></span>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="contact_name"><?php _e('Contact name:', 'jobbin');?></label></th>
				<td><input type="text" class="regular-text" id="contact_name" name="contact_name" size="40" value="<?php echo esc_attr($job['contact_name']);?>" /></td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="contact_email"><?php _e('Contact/application email:', 'jobbin');?></label></th>
				<td>
					<input type="text" class="regular-text" id="contact_email" name="contact_email" size="40" value="<?php echo esc_attr($job['contact_email']);?>" />
					<br/><span class="description"><?php _e('This email address is used by application form on detail job page.', 'jobbin');?></span>
				</td>
			</tr>
			<tr valign="top">
				<th scope="row"><label for="ad_duration"><?php _e('Duration (in days):', 'jobbin');?></label></th>
				<td>
					<input type="text" id="ad_duration" name="ad_duration" value="<?php echo esc_attr($job['ad_duration']);?>" size="3" />
					<strong><?php _e('Expiration date:', 'jobbin');?></strong> <?php echo date($date_format.' '.$time_format, strtotime($job['expired']) + $gmt_offset);?>
					<br /><span class="description"><?php _e('Enter a high number (e.g. 9999) for no expiration.', 'jobbin');?></span>
				</td>
			</tr>
			<tr valign="top">
				<th scope="row"><label for="ad_currency"><?php _e('Currency &amp; price:', 'jobbin');?></label></th>
				<td>
					<select id="ad_currency" name="ad_currency">
						<?php foreach($currencies as $cur => $txt) :?>
							<option value="<?php echo esc_attr($cur);?>" <?php jobbin_selected($job['ad_currency'], $cur);?>><?php echo esc_attr($txt);?></option>
						<?php endforeach;?>
					</select>
					<input type="text" name="ad_price" value="<?php echo esc_attr($job['ad_price']);?>" size="3" />
					<br/><span class="description"><?php _e('Setting the price to 0 means free job listings.', 'jobbin');?></span>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="ad_paid"><?php _e('Paid:', 'jobbin');?></label></th>
				<td>
					<select id="ad_paid" name="ad_paid">
						<option value="0" <?php jobbin_selected($job['ad_paid'], 0) ?>><?php _e('No');?></option>
						<option value="1" <?php jobbin_selected($job['ad_paid'], 1) ?>><?php _e('Yes');?></option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="ad_approved"><?php _e('Approved:', 'jobbin');?></label></th>
				<td>
					<select id="ad_approved" name="ad_approved">
						<option value="0" <?php jobbin_selected($job['ad_approved'], 0) ?>><?php _e('No');?></option>
						<option value="1" <?php jobbin_selected($job['ad_approved'], 1) ?>><?php _e('Yes');?></option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="ad_published"><?php _e('Published:', 'jobbin');?></label></th>
				<td>
					<select id="ad_published" name="ad_published">
						<option value="0" <?php jobbin_selected($job['ad_published'], 0) ?>><?php _e('No');?></option>
						<option value="1" <?php jobbin_selected($job['ad_published'], 1) ?>><?php _e('Yes');?></option>
					</select>
				</td> 
			</tr>
			<tr>
				<th scope="row" valign="top"><label for="txn_id"><?php _e('PayPal Transaction ID:', 'jobbin') ?></th>
				<td><input type="text" class="regular-text" id="txn_id" name="txn_id" size="40" value="<?php echo esc_attr($job['txn_id']) ?>" /></td>
			</tr>
		</table>
		<p>
			<input type="submit" name="Submit" value="<?php echo esc_attr($submit_text) ?>" class="button-primary" />
			<hr /><span class="description">If you receive a <strong>job not added</strong> error message, you probably forgot to fill in one of the fields.
			<br /><strong>All fields are mandatory except where specified.</strong></span>
		</p>
	</form>
	<?php
}

function jobbin_admin_add_job_form($job, $form, $action, $submit_text) {
	if(is_null($job)) {
		$jobbin_options = get_option('jobbin_options');
		$job = array();
		$job['id'] = '';
		$job['posted'] = '';
		$job['modified'] = $job['posted'];
		$job['expired'] = gmdate('Y-m-d H:i:s', time() + intval($jobbin_options['duration']) * 86400);
		$job['title'] = '';
		$job['category'] = $jobbin_options['default_category'];
		$job['description'] = '';
		$job['how_to_apply'] = '';
		$job['company_name'] = '';
		$job['location'] = '';
		$job['company_url'] = '';
		$job['contact_name'] = '';
		$job['contact_email'] = '';
		$job['ad_duration'] = intval($jobbin_options['duration']);
		$job['ad_currency'] = $jobbin_options['currency'];
		$job['ad_price'] = 0;
		$job['ad_paid'] = 0;
		$job['ad_approved'] = 0;
		$job['ad_published'] = 0;
		$job['txn_id'] = '';
	}
	jobbin_admin_job_form($job, $form, $action, $submit_text);
}

function jobbin_admin_edit_job_form($job, $form, $action, $submit_text) {
	jobbin_admin_job_form($job, $form, $action, $submit_text);
}

function jobbin_admin_load_options() {
	if(isset($_POST['action'])) {
		$action = $_POST['action'];
		if($action == 'update') return jobbin_admin_update_options();
	}
}

function jobbin_admin_options() {
	$jobbin_options = get_option('jobbin_options');

	$messages[1] = __('Options updated.', 'jobbin');
	$messages[2] = __('Options not updated.', 'jobbin');

	$form = '<form method="post" action="admin.php?page=jobbin-admin-options&amp;action=update">';
	$action = 'update';
	$heading = __('Options', 'jobbin');
	$submit_text = __('Save Changes');
	?>
	<?php if(isset($_GET['message'])) :?>
		<div id="message" class="updated fade"><p><?php echo $messages[$_GET['message']];?></p></div>
	<?php endif;?>

	<div class="wrap">
		<div id="icon-plugins" class="icon32"></div>
		<h2><?php echo $heading;?></h2>
		<?php jobbin_admin_options_form($jobbin_options, $form, $action, $submit_text);?>
	</div>
	<?php
}

function jobbin_admin_load_uninstall() {
	if(isset($_POST['action'])) {
		$action = $_POST['action'];
		if($action == 'uninstall') return jobbin_uninstall();
		if($action == 'keep') return jobbin_uninstall_keep();
	}
}
function jobbin_admin_uninstall() {
	$heading = __('Uninstall', 'jobbin');
	?>
	<div class="wrap">
		<div id="icon-plugins" class="icon32"></div>
		<h2><?php echo $heading;?></h2>
		<p>Uninstalling <strong>Jobbin</strong> will remove all data associated with this plugin. Are you sure you want to do this?</p>
		<p><strong>WARNING:</strong> Deleting all jobs and categories cannot be undone!</p>
		<form method="post" action="admin.php?page=jobbin-admin-uninstall">
			<p>
				<input type="hidden" name="action" value="uninstall" />
				<input type="submit" class="button-secondary action" value="Uninstall Jobbin and delete all jobs and categories" />
			</p>
		</form>
		<form method="post" action="admin.php?page=jobbin-admin-uninstall">
			<p>
				<input type="hidden" name="action" value="keep" />
				<input type="submit" class="button-secondary action" value="Uninstall Jobbin and keep all jobs and categories" />
			</p>
		</form>
		<p><strong>NOTE:</strong> Some WordPress installations do not automatically deactivate the plugin after uninstalling it. Please check if the plugin is deactivated.</p>
	</div>
<?php
}

function jobbin_get_all_currencies() {
	return array('USD' => 'USD', 'CAD' => 'CAD', 'EUR' => 'EUR', 'GBP' => 'GBP', 'YEN' => 'YEN', 'AUD' => 'AUD', 'NZD' => 'NZD', 'CHF' => 'CHF', 'HKD' => 'HKD', 'SGD' => 'SGD', 'SEK' => 'SEK', 'DKK' => 'DKK', 'PLN' => 'PLN', 'NOK' => 'NOK', 'HUF' => 'HUF', 'CZK' => 'CZK');
}

function jobbin_admin_options_form($jobbin_options, $form, $action, $submit_text) {
	global $wpdb, $wp_version;
	if(!$jobbin_options)
		$jobbin_options = get_option('jobbin_options');
	$currencies = jobbin_get_all_currencies();

	$pid = $jobbin_options['post_id'];

	echo $form;
	?>
	<input type="hidden" name="action" value="<?php echo $action;?>" />

	<table class="form-table boxin">
		<tr><td colspan="2"><h2>Job Board Options</h2></td></tr>
		<tr valign="top">
			<th scope="row"><?php _e('Job board title:', 'jobbin');?></th>
			<td>
				<input type="text" class="regular-text" name="title" value="<?php echo esc_attr($jobbin_options['title']);?>" size="40" />
				<p class="description">This title will show up on your jobs page. Your jobs page has ID <?php echo $pid;?>.</p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('Job posting policy/guide:', 'jobbin');?></th>
			<td>
				<textarea name="description" class="large-text" rows="5" cols="50"><?php echo $jobbin_options['description'];?></textarea>
				<p class="description">This title will show up on your <strong>Add a job</strong> form.</p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('Invite:', 'jobbin');?></th>
			<td>
				<input type="text" class="regular-text" name="invite" value="<?php echo esc_attr($jobbin_options['invite']) ?>" size="40" />
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('Enable new job listings?', 'jobbin');?></th>
			<td>
				<input type="hidden" name="enable_frontend" value="0" />
				<input type="checkbox" id="enable_frontend" name="enable_frontend" value="1" <?php jobbin_checked($jobbin_options['enable_frontend'], 1);?> />
				<p class="description">Checking this will enable the <strong>Add a job</strong> form.</p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('Enable posting to guests?', 'jobbin');?></th>
			<td>
				<input type="hidden" name="enable_guests" value="0" />
				<input type="checkbox" id="enable_guests" name="enable_guests" value="1" <?php jobbin_checked($jobbin_options['enable_guests'], 1);?> />
				<p class="description">Checking this will enable guests to add new jobs.</p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('Duration (in days):', 'jobbin');?></th>
			<td><input type="text" name="duration" value="<?php echo esc_attr($jobbin_options['duration']);?>" size="3" /></td>
		</tr>
	</table>

	<table class="form-table boxin">
		<tr><td colspan="2"><h2>Payment Options</h2></td></tr>
		<tr valign="top">
			<th scope="row"><?php _e('Currency:', 'jobbin') ?></th>
			<td>
				<select name="currency">
					<?php foreach($currencies as $cur => $txt): ?>
						<option value="<?php echo esc_attr($cur);?>" <?php jobbin_selected($jobbin_options['currency'], $cur);?>><?php echo esc_attr($txt);?></option>
					<?php endforeach;?>
				</select>
			</td>
		</tr>
		<tr valign="top">
			<th><?php _e('Full time price:', 'jobbin');?></th>
			<td><input type="text" name="fulltime_price" value="<?php echo esc_attr($jobbin_options['fulltime_price']);?>" size="3" /></td>
		</tr>
		<tr valign="top">
			<th><?php _e('Part time price:', 'jobbin');?></th>
			<td><input type="text" name="parttime_price" value="<?php echo esc_attr($jobbin_options['parttime_price']);?>" size="3" /></td>
		</tr>
		<tr valign="top">
			<th><?php _e('Freelance price:', 'jobbin');?></th>
			<td><input type="text" name="freelance_price" value="<?php echo esc_attr($jobbin_options['freelance_price']);?>" size="3" /></td>
		</tr>
		<tr valign="top">
			<th><?php _e('Internship price:', 'jobbin');?></th>
			<td><input type="text" name="internship_price" value="<?php echo esc_attr($jobbin_options['internship_price']);?>" size="3" /></td>
		</tr>
		<tr valign="top">
			<th><?php _e('Apprenticeship price:', 'jobbin');?></th>
			<td><input type="text" name="apprenticeship_price" value="<?php echo esc_attr($jobbin_options['apprenticeship_price']);?>" size="3" /></td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('PayPal email:', 'jobbin');?></th>
			<td>
				<input type="text" class="code" name="paypal_email" value="<?php echo esc_attr($jobbin_options['paypal_email']);?>" size="40" />
				<p class="description">This email address must match your PayPal email address.</p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('PayPal URL:', 'jobbin');?></th>
			<td>
				<input type="text" class="code" name="paypal_url" value="<?php echo esc_attr($jobbin_options['paypal_url']);?>" size="40" />
				<p><span class="code"><?php _e('This should generally be https://www.paypal.com/cgi-bin/webscr', 'jobbin');?></span></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('PayPal verification URL');?></th>
			<td>
				<input type="text" class="code" name="paypal_verification_url" value="<?php echo esc_attr($jobbin_options['paypal_verification_url']);?>" size="40" />
				<p><span class="code"><?php _e('This should generally be ssl://www.paypal.com:443/cgi-bin/webscr', 'jobbin');?></span></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row"><?php _e('Job listing date format:', 'jobbin');?></th>
			<td>
				<input type="text" class="regular-text" name="date_format" value="<?php echo esc_attr($jobbin_options['date_format']);?>" size="30" />
				<p class="description">Output: <strong><?php echo date($jobbin_options['date_format'], time());?></strong>. <?php _e('<a href="http://codex.wordpress.org/Formatting_Date_and_Time">Documentation on date formatting</a>. Click "Save Changes" to update sample output.');?></p>
			</td>
		</tr>
	</table>

	<table class="form-table boxin">
		<tr><td colspan="2"><h2>Terms, Conditions and Notification Emails</h2></td></tr>
		<tr valign="top">
			<th scope="row" valign="top"><?php _e('Terms:', 'jobbin');?></th>
			<td>
				<textarea name="terms" class="large-text code" rows="5" cols="50"><?php echo esc_attr($jobbin_options['terms']);?></textarea>
				<p class="description"><?php _e('Shown to ad buyers prior to submission.', 'jobbin');?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" valign="top"><?php _e('Payment email:', 'jobbin');?></th>
			<td>
				Subject<br /><input type="text" class="regular-text" name="payment_email_subject" value="<?php echo esc_attr($jobbin_options['payment_email_subject']);?>" size="40" /><br />
				Message<br /><textarea name="payment_email_message" class="large-text code" rows="5" cols="50"><?php echo esc_attr($jobbin_options['payment_email_message']);?></textarea><br />
				<p class="description"><?php _e('Sent to advertisers prior to payment.', 'jobbin');?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" valign="top"><?php _e('Publish email:', 'jobbin');?></th>
			<td>
				Subject<br /><input type="text" class="regular-text" name="publish_email_subject" value="<?php echo esc_attr($jobbin_options['publish_email_subject']);?>" size="40" /><br />
				Message<br /><textarea name="publish_email_message" class="large-text code" rows="5" cols="50"><?php echo esc_attr($jobbin_options['publish_email_message']);?></textarea><br />
				<p class="description"><?php _e('Sent to advertisers when their ad is published.', 'jobbin');?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" valign="top"><?php _e('Email from:', 'jobbin');?></th>
			<td>
				<input type="text" name="email_from_name" value="<?php echo esc_attr($jobbin_options['email_from_name']);?>" size="10" />
				<input type="text" name="email_from" value="<?php echo esc_attr($jobbin_options['email_from']);?>" size="30" />
				<p class="description"><?php _e('Name and email address all outgoing mails are from', 'jobbin');?></p>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" valign="top"><?php _e('Expired ads viewable?', 'jobbin');?></th>
			<td>
				<input type="hidden" name="viewable_expired_ads" value="0" />
				<input type="checkbox" id="viewable_expired_ads" name="viewable_expired_ads" value="1" <?php jobbin_checked($jobbin_options['viewable_expired_ads'], 1);?> />
				<label for="viewable_expired_ads"><?php _e('Checking this will allow your site users to view expired ads', 'jobbin');?></label>
			</td>
		</tr>
		<tr valign="top">
			<th scope="row" valign="top"><?php _e('Enable random job ad in posts?', 'jobbin');?></th>
			<td>
				<input type="hidden" name="enable_random_ad" value="0" />
				<input type="checkbox" id="enable_random_ad" name="enable_random_ad" value="1" <?php jobbin_checked($jobbin_options['enable_random_ad'], 1);?> />
				<label for="enable_random_ad"><?php _e('Checking this will enable random job ad in every post', 'jobbin');?></label>
			</td>
		</tr>
	</table>
	<input type="hidden" name="page_options" value="title,description,invite,enable_frontend,enable_guests,duration,currency,fulltime_price,parttime_price,freelance_price,internship_price,apprenticeship_price,workingfromhome_price,paypal_email,paypal_url,paypal_verification_url,date_format,terms,payment_email_subject,payment_email_message,publish_email_subject,publish_email_message,email_from_name,email_from,viewable_expired_ads,enable_random_ad" />
	<p><input type="submit" name="Submit" class="button-primary" value="<?php echo esc_attr($submit_text);?>" /></p>
	</form>
	<?php
}

function jobbin_admin_update_options() {
	if(jobbin_update_options(stripslashes_deep($_POST)))
		wp_redirect('admin.php?page=jobbin-admin-options&message=1');
	else
		wp_redirect('admin.php?page=jobbin-admin-options&message=2');
}

function jobbin_update_options($options) {
	$jobbin_options = get_option('jobbin_options');
	$page_options = explode(',', $options['page_options']);
	foreach($page_options as $key) {
		if(isset($options[$key]))
			$jobbin_options[$key] = $options[$key];
	}
	return update_option('jobbin_options', $jobbin_options);
}

function jobbin_admin_add_category() {
	if(jobbin_insert_category(stripslashes_deep($_POST)))
		wp_redirect('admin.php?page=jobbin-admin-categories&message=1');
	else
		wp_redirect('admin.php?page=jobbin-admin-categories&message=4');
}

function jobbin_admin_edit_category() {
	$cat_ID = mysql_real_escape_string($_GET['cat_ID']);
	$category = jobbin_get_category($cat_ID);
	$heading = __('Edit Job Category', 'jobbin');
	$submit_text = __('Update Job Category', 'jobbin');
	$form = '<form name="editcat" id="editcat" method="post" action="admin.php?page=jobbin-admin-categories&amp;action=update">';
	$action = 'update';
	?>
	<div class="wrap">
		<h2><?php echo $heading;?></h2>
		<?php jobbin_admin_category_form($category, $form, $action, $submit_text);?>
	</div>
	<?php
}

function jobbin_admin_update_category() {
	$cat_ID = mysql_real_escape_string($_POST['cat_ID']);

	if(jobbin_update_category(stripslashes_deep($_POST)))
		wp_redirect('admin.php?page=jobbin-admin-categories&message=3');
	else
		wp_redirect('admin.php?page=jobbin-admin-categories&message=5');
}

function jobbin_admin_delete_category() {
	$cat_ID = (int) $_GET['cat_ID'];
	$cat_name = jobbin_get_catname($cat_ID);
	$jobbin_options = get_option('jobbin_options');

	if($cat_ID == $jobbin_options['default_category'])
		wp_die(sprintf(__('Can&#8217;t delete the <strong>%s</strong> category: this is the default one'), $cat_name));

	jobbin_delete_category($cat_ID);
	wp_redirect('admin.php?page=jobbin-admin-categories&message=2');
}

function jobbin_admin_add_job() {
	$_POST['ad_paid'] = intval($_POST['ad_price']) == 0 ? 1 : 0;

	if(jobbin_insert_job(stripslashes_deep($_POST)))
		wp_redirect('admin.php?page=jobbin-admin-jobs&message=1');
	else
		wp_redirect('admin.php?page=jobbin-admin-jobs&message=4');
}

function jobbin_admin_edit_job() {
	$job_ID = (int) $_GET['job_ID'];
	$job = jobbin_get_job($job_ID);
	$heading = __('Edit Job', 'jobbin');
	$submit_text = __('Update Job', 'jobbin');
	$form = '<form name="editjob" id="editjob" method="post" action="admin.php?page=jobbin-admin-jobs&amp;action=update">';
	$action = 'update';
	?>
	<div class="wrap">
		<h2><?php echo $heading;?></h2>
		<?php jobbin_admin_edit_job_form($job, $form, $action, $submit_text) ?>
	</div>
	<?php
}

function jobbin_admin_update_job() {
	$job_ID = $_POST['job_ID'];

	if(jobbin_update_job(stripslashes_deep($_POST)))
		wp_redirect('admin.php?page=jobbin-admin-jobs&message=3');
	else
		wp_redirect('admin.php?page=jobbin-admin-jobs&message=5');
}

function jobbin_admin_delete_job() {
	$job_ID = (int) $_GET['job_ID'];

	jobbin_delete_job($job_ID);
	wp_redirect('admin.php?page=jobbin-admin-jobs&message=2');
}

// Data functions
function jobbin_get_category($cat_ID) {
	global $wpdb;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$sql = "SELECT id, name, priority, job_count FROM $table_category WHERE id = '".$cat_ID."' LIMIT 1";
	$category = $wpdb->get_row($sql, ARRAY_A);
	return $category;
}

function jobbin_get_all_categories() {
	global $wpdb;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$sql = "SELECT id, name, priority, job_count FROM $table_category ORDER BY priority ASC, name ASC";
	$categories = $wpdb->get_results($sql, ARRAY_A);
	return $categories;
}

function jobbin_get_catname($cat_ID) {
	global $wpdb;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$sql = "SELECT name FROM $table_category WHERE id = " . intval($cat_ID) . " LIMIT 1";
	$name = $wpdb->get_var($sql);
	return $name;
}

function jobbin_insert_category($category) {
	global $wpdb;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	if(empty($category['name']))
		return false;

	$category['priority'] = intval($category['priority']);

	if(!$category['priority'])
		return false;
	$sql = "INSERT INTO $table_category (name, priority) VALUES ('".$wpdb->escape($category['name'])."', ".$category['priority'].")";
	$result = $wpdb->query($sql);
	return $result;
}

function jobbin_update_category($category) {
	global $wpdb;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	if(empty($category['cat_ID']) || empty($category['name']) || empty($category['priority']))
		return false;

	$category['id'] = intval($category['cat_ID']);
	$category['priority'] = intval($category['priority']);

	if($category['priority'] < 1)
		return false;
	$sql = "UPDATE $table_category SET name = '".$wpdb->escape($category['name'])."', priority = ".$category['priority']." WHERE id = ".$category['id'];
	$result = $wpdb->query($sql);
	return $result;
}

function jobbin_delete_category($cat_ID) {
	global $wpdb;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$sql = "DELETE FROM $table_category WHERE id = ".intval($cat_ID);
	$result = $wpdb->query($sql);
	return $result;
}

function jobbin_get_all_types() {
	return array(
		array('id' => 'fulltime', 'name' => __('Full Time', 'jobbin')),
		array('id' => 'parttime', 'name' => __('Part Time', 'jobbin')),
		array('id' => 'freelance', 'name' => __('Freelance', 'jobbin')),
		array('id' => 'internship', 'name' => __('Internship', 'jobbin')),
		array('id' => 'apprenticeship', 'name' => __('Apprenticeship', 'jobbin')),
		array('id' => 'workingfromhome', 'name' => __('Working from home', 'jobbin'))
	);
}

function jobbin_job_is_viewable($job_ID) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$now = gmdate('Y-m-d H:i:s', time());
	$sql = "SELECT COUNT(id) FROM $table_job WHERE id = ".$job_ID." AND ad_published = 1 AND ad_approved = 1";
	$viewable = $wpdb->get_var($sql);
	return $viewable;
}

function jobbin_get_job($job_ID) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$sql = "SELECT * FROM $table_job WHERE id = ".intval($job_ID)." LIMIT 1";
	$job = $wpdb->get_row($sql, ARRAY_A);
	return $job;
}

function jobbin_get_all_jobs($cat_ID = null, $ad_approved = 1, $ad_paid = 1, $ad_published = 1) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$columns = "$table_job.id, $table_job.posted, $table_job.title, $table_job.company_name, $table_job.location, $table_job.company_url, $table_category.id AS category_id, $table_category.name AS category_name, $table_job.type";
	$now = gmdate('Y-m-d H:i:s', time());
	$categories = jobbin_get_all_categories();
	if($cat_ID) {
		$cat_ID = intval($cat_ID);
		$sql = "SELECT $columns FROM $table_job INNER JOIN $table_category ON $table_job.category = $table_category.id WHERE ad_approved = $ad_approved AND ad_paid = $ad_paid AND ad_published = $ad_published AND (expired > '$now' OR ad_duration = -1) AND $table_category.id = $cat_ID ORDER BY posted DESC";
	}
	else {
		$sql = "SELECT $columns FROM $table_job INNER JOIN $table_category ON $table_job.category = $table_category.id WHERE ad_approved = $ad_approved AND ad_paid = $ad_paid AND ad_published = $ad_published AND (expired > '$now' OR ad_duration = -1) ORDER BY posted DESC";
	}
	$jobs = array();
	$_jobs = $wpdb->get_results($sql, ARRAY_A);
	$__jobs = array();
	foreach($categories as $category) {
		$__jobs[$category['id']]['label'] = $category['name'];
		$__jobs[$category['id']]['listing'] = array();
	}
	if(!empty($_jobs)) {
		foreach($_jobs as $job) {
			$__jobs[$job['category_id']]['listing'][] = $job;
		}
		foreach($__jobs as $cat => $job) {
			if(!empty($job['listing'])) {
				$jobs[$cat] = $job;
			}
		}
	}
	elseif(isset($cat_ID)) {
		$jobs[$cat_ID] = array('label' => jobbin_get_catname($cat_ID), 'listing' => array());
	}
	return $jobs;
}

function jobbin_search_all_jobs($query = '', $ad_approved = 1, $ad_paid = 1, $ad_published = 1) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
	$columns = "$table_job.id, $table_job.posted, $table_job.title, $table_job.company_name, $table_job.location, $table_job.company_url, $table_category.id AS category_id, $table_category.name AS category_name, $table_job.type";
	$now = gmdate('Y-m-d H:i:s', time());
	$categories = jobbin_get_all_categories();
	if($query) {
		$query = $wpdb->escape($query);
		$sql = "SELECT $columns FROM $table_job INNER JOIN $table_category ON $table_job.category = $table_category.id WHERE ad_approved = $ad_approved AND ad_paid = $ad_paid AND ad_published = $ad_published AND (expired > '$now' OR ad_duration = -1) AND ($table_job.title LIKE '%$query%' OR $table_job.description LIKE '%$query%') ORDER BY posted DESC";
	}
	$jobs = array();
	$_jobs = $wpdb->get_results($sql, ARRAY_A);
	$__jobs = array();
	foreach($categories as $category) {
		$__jobs[$category['id']]['label'] = $category['name'];
		$__jobs[$category['id']]['listing'] = array();
	}
	if(!empty($_jobs)) {
		foreach($_jobs as $job) {
			$__jobs[$job['category_id']]['listing'][] = $job;
		}
		foreach($__jobs as $cat => $job) {
			if(!empty($job['listing'])) {
				$jobs[$cat] = $job;
			}
		}
	}
	elseif(isset($cat_ID)) {
		$jobs[$cat_ID] = array('label' => jobbin_get_catname($cat_ID), 'listing' => array());
	}
	return $jobs;
}

function jobbin_valid_job($job) {
	if(empty($job['title'])) 			return false;
	if(!intval($job['category'])) 		return false;
	if(empty($job['description'])) 		return false;
	if(empty($job['how_to_apply'])) 	return false;
	if(empty($job['location'])) 		return false;
	if(empty($job['company_name'])) 	return false;
	if(empty($job['contact_name'])) 	return false;
	if(empty($job['contact_email'])) 	return false;
	return true;
}

function jobbin_insert_job($job) {
	global $wpdb;
	$jobbin_options = get_option('jobbin_options');
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	if(!jobbin_valid_job($job))
		return false;
	$now = time();

	$job['posted'] 			= $job['modified'] = $wpdb->escape(gmdate('Y-m-d H:i:s', $now));
	$job['expired'] 		= $wpdb->escape(gmdate('Y-m-d H:i:s', $now + intval($job['ad_duration']) * 86400));
	$job['title'] 			= $wpdb->escape($job['title']);
	$job['category'] 		= intval($job['category']);
	$job['type'] 			= $wpdb->escape($job['type']);
	$job['description'] 	= $wpdb->escape($job['description']);
	$job['how_to_apply'] 	= $wpdb->escape($job['how_to_apply']);
	$job['location'] 		= $wpdb->escape($job['location']);
	$job['company_name'] 	= $wpdb->escape($job['company_name']);
	$job['company_url'] 	= $wpdb->escape($job['company_url']);
	$job['contact_name'] 	= $wpdb->escape($job['contact_name']);
	$job['contact_email'] 	= $wpdb->escape($job['contact_email']);
	$job['ad_duration'] 	= $wpdb->escape($job['ad_duration']);
	$job['ad_currency'] 	= $wpdb->escape($job['ad_currency']);
	$job['ad_price'] 		= $wpdb->escape($job['ad_price']);
	$job['ad_paid'] 		= $wpdb->escape($job['ad_paid']);
	$job['ad_approved'] 	= $wpdb->escape($job['ad_approved']);
	$job['ad_published'] 	= $wpdb->escape($job['ad_published']);
	$job['txn_id'] 			= $wpdb->escape(trim($job['txn_id']));
	$job['ip_address'] 		= $wpdb->escape(preg_replace('/[^0-9., ]/', '', $_SERVER['REMOTE_ADDR']));

	$sql = "INSERT INTO $table_job (`posted`, `modified`, `expired`, `title`, `category`, `type`, `description`, `how_to_apply`, `company_name`, `location`, `company_url`, `contact_name`, `contact_email`, `ip_address`, `ad_approved`, `ad_paid`, `ad_duration`, `ad_currency`, `ad_price`, `ad_published`".($job['txn_id']?', `txn_id`':null).")
			VALUES ('".$job['posted']."', '".$job['modified']."', '".$job['expired']."', '".$job['title']."', ".$job['category'].", '".$job['type']."', '".$job['description']."','".$job['how_to_apply']."', '".$job['company_name']."', '".$job['location']."', '".$job['company_url']."', '".$job['contact_name']."', '".$job['contact_email']."', '".$job['ip_address']."', ".$job['ad_approved'].", ".$job['ad_paid'].", ".$job['ad_duration'].", '".$job['ad_currency']."', ".$job['ad_price'].", ".$job['ad_published'].($job['txn_id']?", '".$job['txn_id']."'":null).")";
	$result = $wpdb->query($sql);
	if($result) {
		$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
		$sql = "UPDATE $table_category SET job_count = job_count + 1 WHERE id = ".$job['category'];
		$wpdb->query($sql);
	}
	return $result;
}

function jobbin_update_job($job) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	if(!jobbin_valid_job($job))
		return false;

	$job['id'] 				= $job['job_ID'];
	$job['modified'] 		= $wpdb->escape(gmdate('Y-m-d H:i:s', time()));
	$job['title'] 			= $wpdb->escape($job['title']);
	$job['category'] 		= intval($job['category']);
	$job['type'] 			= $wpdb->escape($job['type']);
	$job['description'] 	= $wpdb->escape($job['description']);
	$job['how_to_apply'] 	= $wpdb->escape($job['how_to_apply']);
	$job['location'] 		= $wpdb->escape($job['location']);
	$job['company_name'] 	= $wpdb->escape($job['company_name']);
	$job['company_url'] 	= $wpdb->escape($job['company_url']);
	$job['contact_name'] 	= $wpdb->escape($job['contact_name']);
	$job['contact_email'] 	= $wpdb->escape($job['contact_email']);
	$job['ad_duration'] 	= $wpdb->escape($job['ad_duration']);
	$job['ad_currency'] 	= $wpdb->escape($job['ad_currency']);
	$job['ad_price'] 		= $wpdb->escape($job['ad_price']);
	$job['ad_paid'] 		= $wpdb->escape($job['ad_paid']);
	$job['ad_approved'] 	= $wpdb->escape($job['ad_approved']);
	$job['ad_published'] 	= $wpdb->escape($job['ad_published']);
	$job['txn_id'] 			= $wpdb->escape(trim($job['txn_id']));

	$old = jobbin_get_job($job['job_ID']);
	$sql = "UPDATE $table_job SET 
        `modified` = '".$job['modified']."', `title` = '".$job['title']."', `category` = ".$job['category'].", `type` = '".$job['type']."', `description` = '".$job['description']."',
        `how_to_apply` = '".$job['how_to_apply']."', `company_name` = '".$job['company_name']."', `location` = '".$job['location']."', 
        `company_url` = '".$job['company_url']."', `contact_name` = '".$job['contact_name']."', `contact_email` = '".$job['contact_email']."', `ad_duration` = ".$job['ad_duration'].", `ad_currency` = '".$job['ad_currency']."', `ad_price` = ".$job['ad_price'].", `ad_paid` = ".$job['ad_paid'].", `ad_approved` = ".$job['ad_approved'].", `ad_published` = ".$job['ad_published'].($job['txn_id']?", `txn_id` = '".$job['txn_id']."'":null)." WHERE `id` = ".$job['id'];
	$result = $wpdb->query($sql);
	if($result and ($old['category'] != $job['category'])) {
		$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
		$wpdb->query("UPDATE $table_category SET job_count = job_count - 1 WHERE job_count > 0 AND id = ".intval($old['category']));
		$wpdb->query("UPDATE $table_category SET job_count = job_count + 1 WHERE id = ".$job['category']);
	}
	return $result;
}

function jobbin_delete_job($job_ID) {
	global $wpdb;
	$job = jobbin_get_job($job_ID);
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$sql = "DELETE FROM $table_job WHERE id = ".intval($job_ID);
	$result = $wpdb->query($sql);
	if($result) {
		$table_category = $wpdb->prefix.JOBBIN_CATEGORY;
		$sql = "UPDATE $table_category SET job_count = job_count - 1 WHERE id = ".$job['category'];
		$wpdb->query($sql);
	}
	return $result;
}

// PayPal functions
function jobbin_paypal_form_quick($jID, $jTitle, $price) {
	$jobbin_options = get_option('jobbin_options');
	ob_start();
	?>
	<form method="post" action="<?php echo esc_attr($jobbin_options['paypal_url']);?>" name="pp_quick" id="pp_quick">
		<input type="hidden" name="cmd" value="_xclick" />
		<input type="hidden" name="business" value="<?php echo esc_attr($jobbin_options['paypal_email']);?>" />
		<input type="hidden" name="item_name" value="<?php echo 'JOB: '.$jTitle;?>" />
		<input type="hidden" name="item_number" value="<?php echo $jID;?>" />
		<input type="hidden" name="amount" value="<?php echo $price;?>" />
		<input type="hidden" name="no_shipping" value="2" />
		<input type="hidden" name="no_note" value="1" />
		<input type="hidden" name="currency_code" value="<?php echo esc_attr($jobbin_options['currency']);?>" />
		<input type="hidden" name="return" value="<?php echo esc_attr(jobbin_get_permalink('action=paypal-return'));?>" />
		<input type="hidden" name="notify_url" value="<?php echo esc_attr(jobbin_get_permalink());?>" />
		<input type="hidden" name="rm" value="1" />
		<input type="hidden" name="bn" value="PP-BuyNowBF" />
		<input type="image" src="https://www.paypal.com/en_US/i/btn/x-click-but23.gif" border="0" name="submit" onclick="document.forms['pp_quick'].submit();return false" alt="Make payments with PayPal - it's fast, free and secure!" />
		<img alt="" border="0" src="https://www.paypal.com/en_US/i/scr/pixel.gif" width="1" height="1" />
	</form>
	<?php
	$contents = ob_get_contents();
	ob_end_clean();
	return $contents;
}

function jobbin_paypal_form($job) {
	$jobbin_options = get_option('jobbin_options');
	ob_start();
	?>
	<form method="post" action="<?php echo esc_attr($jobbin_options['paypal_url']);?>">
		<input type="hidden" name="cmd" value="_xclick" />
		<input type="hidden" name="business" value="<?php echo esc_attr($jobbin_options['paypal_email']);?>" />
		<input type="hidden" name="item_name" value="<?php echo esc_attr(sprintf(__('Job: %s', 'jobbin'), $job['title']));?>" />
		<input type="hidden" name="item_number" value="<?php echo esc_attr($job['id']);?>" />
		<input type="hidden" name="amount" value="<?php echo esc_attr($job['ad_price']);?>" />
		<input type="hidden" name="no_shipping" value="2" />
		<input type="hidden" name="no_note" value="1" />
		<input type="hidden" name="currency_code" value="<?php echo esc_attr($job['ad_currency']);?>" />
		<input type="hidden" name="return" value="<?php echo esc_attr(jobbin_get_permalink('action=paypal-return'));?>" />
		<input type="hidden" name="notify_url" value="<?php echo esc_attr(jobbin_get_permalink());?>" />
		<input type="hidden" name="rm" value="1" />
		<input type="hidden" name="bn" value="PP-BuyNowBF" />
		<input type="image" src="https://www.paypal.com/en_US/i/btn/x-click-but23.gif" border="0" name="submit" alt="Make payments with PayPal - it's fast, free and secure!" />
		<img alt="" border="0" src="https://www.paypal.com/en_US/i/scr/pixel.gif" width="1" height="1" />
	</form>
	<?php
	$contents = ob_get_contents();
	ob_end_clean();
	return $contents;
}

function jobbin_paypal_ipn() {
	$request[] = 'cmd=_notify-validate';
	foreach($_POST as $key => $value) {
		$request[] = urlencode(stripslashes($key)).'='.urlencode(stripslashes($value));
	}
	$job = jobbin_get_job($_POST['item_number']);
	if(!$job) {
		header('HTTP/1.1 200 OK');
		exit;
	}
	$jobbin_options = get_option('jobbin_options');
	$paypal_url = $jobbin_options['paypal_url'];
	list($header, $response) = jobbin_paypal_verify(implode('&', $request), array(), 30);
	// make sure we send HTTP 200 response
	header('HTTP/1.1 200 OK');
	if($response !== false) {
		// check for validation
		if(strcmp(strtoupper($response), 'VERIFIED') == 0) {
			// check that the payment_status is Completed
			if($_POST['payment_status'] != 'Completed') {
				exit;
			}
			// check that txn_id has not been previously processed
			if(jobbin_txn_id_exists($_POST['txn_id'])) {
				exit;
			}
			// check that receiver_email is your primary PayPal email
			if($_POST['receiver_email'] != $jobbin_options['paypal_email']) {
				exit;
			}
			// check that item_number, payment_amount and payment_currency are all correct
			if($_POST['item_number'] != $job['id'] or $_POST['mc_gross'] != $job['ad_price'] or $_POST['mc_currency'] != $job['ad_currency']) {
				exit;
			}

			// set job ad to "paid" and update its txn_id in one go
			jobbin_set_paid($job, $_POST['txn_id']);

			// send email to advertiser
			jobbin_send_publish_email($job);

			// done!
			exit;
		}
		elseif(strcmp(strtoupper($response), 'INVALID') == 0) {
			header('HTTP/1.1 200 OK');
			exit;
		}
	}
	else {
		exit;
	}
}

function jobbin_paypal_return() {
	ob_start();
	?>
	<p><?php _e('Thank you for your purchase. Your payment is being processed.', 'jobbin');?></p>
	<?php
	$contents = ob_get_contents();
	ob_end_clean();
	return $contents;
}

function jobbin_txn_id_exists($txn_id) {
	global $wpdb;
	$table_job = $wpdb->prefix.JOBBIN_JOB;
	$txn_id = $wpdb->escape($txn_id);
	$total = $wpdb->get_var("SELECT COUNT(txn_id) AS total FROM $table_job WHERE txn_id = '$txn_id'");
	return $total != '0' ? true : false;
}

// Mail functions
function jobbin_send_payment_email($job) {
	if(!JOBBIN_SEND_EMAIL) return true;
	$jobbin_options = get_option('jobbin_options');
	$to = $job['contact_email'];
	$subject = $jobbin_options['payment_email_subject'];
	$message = $jobbin_options['payment_email_message'];

	$subject = preg_replace('/%job_title%/', $job['title'], $subject);
	$subject = preg_replace('/%contact_name%/', $job['contact_name'], $subject);

	$message = preg_replace('/%job_title%/', $job['title'], $message);
	$message = preg_replace('/%payment_url%/', jobbin_get_permalink('action=paypal&job_id=' . $job['id']), $message);
	$message = preg_replace('/%payment_link%/', '<a href="' . jobbin_get_permalink('job_id=' . $job['id']) . '">' . esc_attr($job['title']) . '</a>', $message);
	$message = preg_replace('/%contact_name%/', $job['contact_name'], $message);
	$message = preg_replace('/%ad_expiration%/', $job['expired'], $message);

	add_filter('wp_mail_from_name', 'jobbin_mail_from_name');
	add_filter('wp_mail_from', 'jobbin_mail_from');
	$sent = wp_mail($to, $subject, $message, $headers = '');
	remove_filter('wp_mail_from_name', 'jobbin_mail_from_name');
	remove_filter('wp_mail_from', 'jobbin_mail_from');
	return $sent;
}

function jobbin_send_publish_email($job) {
	if(!JOBBIN_SEND_EMAIL) return true;
	$jobbin_options = get_option('jobbin_options');
	$to = $job['contact_email'];
	$subject = $jobbin_options['publish_email_subject'];
	$message = $jobbin_options['publish_email_message'];

	$subject = preg_replace('/%job_title%/', $job['title'], $subject);
	$subject = preg_replace('/%contact_name%/', $job['contact_name'], $subject);

	$message = preg_replace('/%job_title%/', $job['title'], $message);
	$message = preg_replace('/%ad_url%/', jobbin_get_permalink('job_id='.$job['id']), $message);
	$message = preg_replace('/%ad_link%/', '<a href="'.jobbin_get_permalink('job_id='.$job['id']).'">'.esc_attr($job['title']).'</a>', $message);
	$message = preg_replace('/%contact_name%/', $job['contact_name'], $message);
	$message = preg_replace('/%ad_expiration%/', $job['expired'], $message);

	add_filter('wp_mail_from_name', 'jobbin_mail_from_name');
	add_filter('wp_mail_from', 'jobbin_mail_from');
	$sent = wp_mail($to, $subject, $message, $headers = '');
	remove_filter('wp_mail_from_name', 'jobbin_mail_from_name');
	remove_filter('wp_mail_from', 'jobbin_mail_from');
	return $sent;
}

function jobbin_mail_from_name($value) {
	$jobbin_options = get_option('jobbin_options');
	return $jobbin_options['email_from_name'];
}

function jobbin_mail_from($value) {
	$jobbin_options = get_option('jobbin_options');
	return $jobbin_options['email_from'];
}

// Widget functions
function jobbin_widget_init() {
	if(!function_exists('wp_register_sidebar_widget') or !function_exists('wp_register_widget_control')) {
		return;
	}
	else {
		wp_register_sidebar_widget('jobbin', 'Jobbin List', 'jobbin_widget', array('classname' => 'widget_jobbin', 'description' => 'Jobbin widget for listing current jobs and adding jobs'));
		wp_register_widget_control('jobbin', 'Jobbin', 'jobbin_widget_control');
		wp_register_sidebar_widget('jobbin_search', 'Jobbin Search', 'jobbin_widget_search', array('classname' => 'widget_jobbin', 'description' => 'Jobbin widget for searching the database'));
	}
}
add_action('plugins_loaded', 'jobbin_widget_init');

function jobbin_widget($args) {
	extract($args);
	$jobbin_options = get_option('jobbin_options');

	$widget_title = empty($jobbin_options['widget_title']) ? 'Jobbin' : esc_attr($jobbin_options['widget_title']);
	$widget_invite = empty($jobbin_options['widget_invite']) ? 'Add a job' : esc_attr($jobbin_options['widget_invite']);

	$categories = jobbin_get_all_categories();
	$out = '<li><a href="'.jobbin_get_permalink().'">All Jobs</a></li>';
	foreach($categories as $cat) {
		$out .= '<li><a href="'.jobbin_get_permalink('cat='.esc_attr($cat['id'])).'">'.esc_attr($cat['name']).'</a></li>';
	}
	if($jobbin_options['enable_frontend'])
		$out .= '<p><a href="'.jobbin_get_permalink('action=postjob').'">'.$widget_invite.'</a></p>';
	echo $before_widget.$before_title.$widget_title.$after_title;
	?>
	<ul>
		<?php echo $out;?>
	</ul>
	<?php
	echo $after_widget;
}
function jobbin_widget_control() {
	$jobbin_options = get_option('jobbin_options');
	if(isset($_POST['jobbin-submit'])) {
		$new_options = $jobbin_options;
		$widget_title = strip_tags(stripslashes($_POST['jobbin-title']));
		if($jobbin_options['widget_title'] != $widget_title) {
			$new_options['widget_title'] = $widget_title;
		}
		$invite = strip_tags(stripslashes($_POST['jobbin-invite']));
		if($jobbin_options['widget_invite'] != $invite) {
			$new_options['widget_invite'] = $invite;
		}
		if($jobbin_options != $new_options) {
			$jobbin_options = $new_options;
			update_option('jobbin_options', $jobbin_options);
		}
	}
	?>
	<p><label for="jobbin-title">Title:</label> <input id="jobbin-title" class="widefat" name="jobbin-title" type="text" value="<?php echo esc_attr($jobbin_options['widget_title']);?>" /></p>
	<p><label for="jobbin-invite">Invite:</label> <input id="jobbin-invite" class="widefat" name="jobbin-invite" type="text" value="<?php echo esc_attr($jobbin_options['widget_invite']);?>" /></p>
	<input type="hidden" id="jobbin-submit" name="jobbin-submit" value="1" />
<?php
}
function jobbin_widget_search($args) {
	extract($args);
	$search_url = get_option('home');
	$jobbin_options = get_option('jobbin_options');

	echo $before_widget;
	echo $before_title.'Find a job'.$after_title;
	?>
	<form method="get" action="<?php echo esc_attr($search_url);?>">
		<div>
			<small>Enter a keyword and search our jobs database.</small>
		</div>
		<div>
			<input type="hidden" value="<?php echo esc_attr($jobbin_options['post_id']);?>" name="page_id" />
			<input type="text" name="search" id="search" />
			<input type="submit" value="Go" id="jobsearchsubmit" />
		</div>
	</form>
	<?php echo $after_widget;?>
<?php
}

function jobbin_rss($args) {
	$jobbin_rss_url = jobbin_get_permalink('jobfeed=rss2');
	?>
	<li><a href="<?php echo esc_attr($jobbin_rss_url);?>" title="Syndicate this site using RSS 2.0">Jobs <abbr title="Really Simple Syndication">RSS</abbr></a></li>
<?php
}
add_action('wp_meta', 'jobbin_rss');
?>
