<?php
use MatthiasWeb\RealMediaLibrary\attachment;
use MatthiasWeb\RealMediaLibrary\general;
use MatthiasWeb\RealMediaLibrary\metadata;
use MatthiasWeb\RealMediaLibrary\order;
use MatthiasWeb\RealMediaLibrary\folder;

defined( 'ABSPATH' ) or die( 'No script kiddies please!' );

/*
 * Meta tags for folders.
 * 
 * @table wp_realmedialibrary_meta
 * $wpdb->realmedialibrarymeta
 * 
 * (C) add_media_folder_meta(...)
 * (R) get_media_folder_meta(...)
 * (U) update_media_folder_meta(...)
 * (D) delete_folder_meta(...)
 * 
 * delete_media_folder_meta_by_key(...): Delete everything from folder meta matching meta key.
 * 
 * Here you can use the default meta data hooks like:
 *  add_{$meta_type}_meta
 *  => add_realmedialibrary_meta
 * 
 * @see RML_Meta
 *         metadata\Meta:content_general
 *         metadata\Meta:save_general
 * @see assets/js/meta.js
 * 
 * PREDEFINED META KEYS:
 *  description
 *  coverImage
 */

if (!function_exists('get_media_folder_meta')) {
    /**
     * Retrieve folder meta field for a folder.
     *
     * @param int $folder_id Folder ID.
     * @param string $key The meta key to retrieve. By default, returns data for all keys.
     * @param boolean $single Whether to return a single value. Default false.
     * @returns mixed Will be an array if $single is false. Will be value of meta data field if $single is true.
     */
    function get_media_folder_meta( $folder_id, $key = '', $single = false ) {
        metadata\Meta::getInstance(); // Necessery checks to prepare metas
        return get_metadata('realmedialibrary', _wp_rml_meta_fix_absint($folder_id), $key, $single);
    }
    
    //var_dump(get_media_folder_meta(108, "test", true));
}

if (!function_exists('add_media_folder_meta')) {
    /**
     * Add meta data field to a folder.
     *
     * Folder meta data is called "Custom Fields" on the Administration Screen.
     *
     * @param int $folder_id Folder ID.
     * @param string $meta_key Metadata name.
     * @param mixed $meta_value Metadata value. Must be serializable if non-scalar.
     * @param boolean $unique Whether the same key should not be added.
     * @returns int|false Meta ID on success, false on failure.
     */
    function add_media_folder_meta( $folder_id, $meta_key, $meta_value, $unique = false ) {
        metadata\Meta::getInstance(); // Necessery checks to prepare metas
        return add_metadata('realmedialibrary', _wp_rml_meta_fix_absint($folder_id), $meta_key, $meta_value, $unique);
    }
}

if (!function_exists('update_media_folder_meta')) {
    /**
     * Update folder meta field based on folder ID.
     *
     * Use the $prev_value parameter to differentiate between meta fields with the
     * same key and folder ID.
     *
     * If the meta field for the folder does not exist, it will be added.
     *
     * @param int $folder_id Folder ID.
     * @param string $meta_key Metadata key.
     * @param mixed $meta_value Metadata value. Must be serializable if non-scalar.
     * @param mixed $prev_value Previous value to check before removing.
     * @returns int|bool Meta ID if the key didn't exist, true on successful update, false on failure.
     */
    function update_media_folder_meta( $folder_id, $meta_key, $meta_value, $prev_value = '' ) {
        metadata\Meta::getInstance(); // Necessery checks to prepare metas
        return update_metadata('realmedialibrary', _wp_rml_meta_fix_absint($folder_id), $meta_key, $meta_value, $prev_value);
    }
    
    //var_dump(update_media_folder_meta(108, "test", "1234"));
}

if (!function_exists('delete_media_folder_meta')) {
    /**
     * Remove metadata matching criteria from a folder.
     *
     * You can match based on the key, or key and value. Removing based on key and
     * value, will keep from removing duplicate metadata with the same key. It also
     * allows removing all metadata matching key, if needed.
     *
     * @param int $folder_id Folder ID.
     * @param string $meta_key Metadata name.
     * @param mixed $meta_value Metadata value. Must be serializable if non-scalar.
     * @returns boolean True on success, false on failure.
     */
    function delete_media_folder_meta( $folder_id, $meta_key, $meta_value = '' ) {
        metadata\Meta::getInstance(); // Necessery checks to prepare metas
        return delete_metadata('realmedialibrary', _wp_rml_meta_fix_absint($folder_id), $meta_key, $meta_value);
    }
    //var_dump(delete_media_folder_meta(108, "test"));
}

if (!function_exists('delete_media_folder_meta_by_key')) {
    /**
     * Delete everything from folder meta matching meta key.
     * 
     * @param string $folder_meta_key Key to search for when deleting.
     * @returns boolean Whether the post meta key was deleted from the database.
     */
    function delete_media_folder_meta_by_key( $folder_meta_key ) {
        metadata\Meta::getInstance(); // Necessery checks to prepare metas
        return delete_metadata( 'realmedialibrary', null, $folder_meta_key, '', true );
    }
}

if (!function_exists('truncate_media_folder_meta')) {
    /**
     * Remove all metas of a folder. Use this with caution!!
     * 
     * @param int $folder_id Folder ID
     * @returns int result of $wpdb->query
     */
    function truncate_media_folder_meta($folder_id) {
        metadata\Meta::getInstance(); // Necessery checks to prepare metas
        global $wpdb;
        
        $sql = $wpdb->prepare("DELETE FROM " . $wpdb->realmedialibrarymeta . " WHERE realmedialibrary_id=%d", _wp_rml_meta_fix_absint($folder_id));
        return $wpdb->query($sql);
    }
    
    //var_dump(truncate_media_folder_meta(108));
}

if (!function_exists('add_rml_user_settings_box')) {
    /**
     * Add a visible content to the general user settings dialog.
     * 
     * @param string $name Unique name for this meta box
     * @param IUserSettings $obj The object which implements IUserSettings
     * @param boolean $hasScripts boolean Load the resources if exists
     * @param int $priority Priority for actions and filters
     * @returns boolean
     */
    function add_rml_user_settings_box($name, $obj, $hasScripts = false, $priority = 10) {
        if (!metadata\Meta::getInstance()->add($name, $obj)) {
            return false;
        }
        
        add_filter('RML/User/Settings/Content', array($obj, 'content'), $priority, 2);
        add_filter('RML/User/Settings/Save', array($obj, 'save'), $priority, 2);
        
        if ($hasScripts) {
            add_action("RML/Backend/Scripts/MediaLibrary", array($obj, "scripts"), $priority);
        }
        return true;
    }
}

if (!function_exists('add_rml_meta_box')) {
    /**
     * Add a visible content to the folder details dialog.
     * 
     * @param string $name Unique name for this meta box
     * @param IMetadata $obj The object which implements IMetadata
     * @param boolean $hasScripts boolean Load the resources if exists
     * @param int $priority Priority for actions and filters
     * @returns boolean
     */
    function add_rml_meta_box($name, $obj, $hasScripts = false, $priority = 10) {
        if (!metadata\Meta::getInstance()->add($name, $obj)) {
            return false;
        }
        
        add_filter('RML/Folder/Meta/Content', array($obj, 'content'), $priority, 2);
        add_filter('RML/Folder/Meta/Save', array($obj, 'save'), $priority, 2);
        
        if ($hasScripts) {
            add_action("RML/Backend/Scripts/MediaLibrary", array($obj, "scripts"), $priority);
        }
        return true;
    }
}

if (!function_exists('_wp_rml_meta_fix_absint')) {
    /**
     * FIX absint() in wordpress
     * 
     * @internal
     */
    function _wp_rml_meta_fix_absint($folder_id) {
        return $folder_id == -1 ? 100000000000 : $folder_id;
    }
}