<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'ST_Reviews_Admin' ) ) :

/**
 * SixthType Reviews Admin
 *
 * A class responsible for loading admin scripts, styles
 * and all plugin admin related tasks.
 * 
 * @version  1.0
 * @since 	 1.0
 * @package  SixthType-Reviews/Includes
 * @author 	 SixthType <hello@sixthtype.com>
 */
class SixthType_Reviews_Admin {

	/**
	 * ADMIN_NONCE_ACTION
	 *
	 * @access 	public
	 * @since 	1.0
	 * @var 	string
	 */
	const ADMIN_NONCE_ACTION = 'st_reviews_admin_nonce_action';
	
	/**
	 * ADMIN_NONCE_ID
	 *
	 *  
	 * @access 	public
	 * @since 	1.0
	 * @var 	string
	 */
	const ADMIN_NONCE_ID = 'st_reviews_admin_nonce_id';

	/**
	 * Plugin Settings Page ID
	 * 
	 * @access  private
	 * @since  	1.0
	 * @var 	string
	 */
	private $page_id = 'sixthtype-reviews';

	/**
	 * Constructor
	 *
	 * @access  public
	 * @version 1.0
	 * @since  	1.0
	 */
	public function __construct() {

		add_action( 'admin_enqueue_scripts', array( &$this, 'load_scripts' ) );
		add_action( 'admin_print_styles', array( &$this, 'load_styles' ) );
		add_action( 'admin_footer', array( &$this, 'print_templates' ) );
		add_action( 'admin_footer', array( &$this, 'add_notification_view') );
		add_action( 'admin_menu', array( &$this, 'add_menu_page' ) );
	}

	/**
	 * Load Scripts
	 *
	 * @access 	public
	 * @version 1.0
	 * @since 	1.0
	 */
	public function load_scripts() {

		global $current_screen;
		
		if( ! $this->is_plugin_page() ) {
		
			return;
		}
		
		// load settings page script only if we are on settings page
		if( $current_screen->id == 'plugins_page_' . $this->page_id ) {
		
			wp_enqueue_script( 
				'st-reviews-page-js',
				ST_REVIEWS_URI . 'admin/js/st-reviews-admin-page.js',
				array( 'jquery' ),
				'1.0',
				true
			);
		}

		wp_enqueue_media();
		wp_enqueue_script( 'underscore' );
		wp_enqueue_script( 'backbone' );
		wp_enqueue_script( 'jquery-ui-core' );
		wp_enqueue_script( 'jquery-ui-sortable' );
		wp_enqueue_script( 'jquery-ui-slider' );

		wp_enqueue_script(
			'jquery-easing',
			ST_REVIEWS_URI . 'admin/js/jquery.easing.1.3.js',
			array( 'jquery' ),
			'1.3',
			true
		);

		wp_enqueue_script(
			'jquery-colorpicker',
			ST_REVIEWS_URI . 'admin/js/colorpicker.js',
			array( 'jquery' ),
			'1.0',
			true
		);

		wp_enqueue_script(
			'stfr-admin-js',
			ST_REVIEWS_URI . 'admin/js/sixthtype-framework-admin.js',
			array( 'jquery', 'jquery-colorpicker', 'jquery-easing', 'underscore' ),
			'1.0',
			true
		);

		wp_enqueue_script(
			'st-reviews-js',
			ST_REVIEWS_URI . 'admin/js/st-reviews-admin.js',
			array( 'jquery', 'backbone', 'stfr-admin-js' ),
			'1.0',
			true
		);

		// localize scripts
		$localized = array( 
			'nonce' => wp_create_nonce( self::ADMIN_NONCE_ACTION ),
			'short_criteria' => __( 'Criteria name is too short.', 'sixthtype-reviews' ),
			'dup_criteria' => __( 'A review criteria with this name already exists!', 'sixthtype-reviews' ),
			'short_criteria_field' => __( 'Rating field name is too short.', 'sixthtype-reviews' ),
			'dup_field' => __( 'A rating field with this name already exists!', 'sixthtype-reviews' ),
		);
		
		$localized_framework = array( 
			'no_gallery_slides' => __( 'No images selected.', 'sixthtype-reviews' )
		);
		
		wp_localize_script( 'st-reviews-js', 'STRW_PAGE_l10n', $localized );		
		wp_localize_script( 'stfr-admin-js', 'STFR_l10n', $localized_framework );

	}

	/**
	 * Load Styles
	 * 
	 * @access 	public
	 * @version 1.0
	 * @since  	1.0
	 */
	public function load_styles() {

		if( ! $this->is_plugin_page() ) {
			
			return;
		}

		wp_enqueue_style(
			'st-reviews-admin-css',
			ST_REVIEWS_URI . 'admin/css/st-reviews-admin-style.css'
		);

		wp_enqueue_style(
			'stfr-admin-css',
			ST_REVIEWS_URI . 'admin/css/sixthtype-framework-admin-style.css'
		);
		
	}

	/**
	 * Print Templates
	 *
	 * Output Backbone views templates
	 * 
	 * @access 	public
	 * @version 1.0
	 * @since 	1.0
	 */
	public function print_templates() {

		if( ! $this->is_plugin_page() ) {
			
			return;
		}

		require_once ( ST_REVIEWS_ROOT . 'admin/templates/templates.php' );
	}

	/**
	 * Add Notification View
	 *
	 * Outputs notification view in admin footer.
	 * Only in plugin settings page.
	 * 
	 * @access  public
	 * @version 1.0
	 * @since  	1.0
	 */
	public function add_notification_view() {

		global $current_screen;
		
		if( $current_screen->id != 'plugins_page_' . $this->page_id || 
			( defined( 'ST_FRAMEWORK_VERSION' ) && 
			  class_exists( 'SixthType_Framework' ) ) ) {
			
			return;
		}

		echo '<div id="stfr_notification_view">
				<span class="content"></span>
				</div>';
	}

	/**
	 * Add Plugin Menu Page
	 * 
	 * @access  public
	 * @version 1.0
	 * @since   1.0
	 */
	public function add_menu_page() {

		add_plugins_page( 
			'SixthType Reviews',
			'SixthType Reviews',
			'manage_options',
			$this->page_id,
			array( &$this, 'settings_page' )
		);

	}

	/**
	 * Plugin Settings Page
	 *
	 * @access  public
	 * @version 1.0
	 * @since   1.0
	 */
	public function settings_page() {

		require_once( ST_REVIEWS_ROOT . 'admin/sixthtype-reviews-settings-page.php' );
	}

	/**
	 * Is Plugin Page
	 *
	 * Returns true if current page is plugins settings page and 
	 * we are on post page that supports reviews
	 * 
	 * @access   public
	 * @version  1.0
	 * @since    1.0
	 * @return   boolean
	 */
	public function is_plugin_page() {

		global $current_screen;
		
		$reviews = SixthType_Reviews::get_instance();
		$post_types = $reviews->get_review_post_types();

		if( $current_screen->id != 'plugins_page_' . $this->page_id &&
			! in_array( $current_screen->post_type, $post_types ) ) {
			
			return false;
		}

		return true;
	}
}

endif; // end class exists check

?>