/*--------------------------------------
 * SixthType Reviews
 *
 * @version  1.0
 --------------------------------------*/

/**
 * Document Ready
 * 
 */
jQuery(document).ready( function($){

	window.ST_REVIEWS = {};

	/**
	 * Defaults
	 * 
	 */
	ST_REVIEWS.defaults = {
		edit_frame: null
	};

	/**
	 * Templates
	 * 
	 */
	ST_REVIEWS.templates = {
		criteria: $('#st-reviews-tmpl-criteria').html(),
		noCriteria: $('#st-reviews-tmpl-no-criteria').html(),
		editCriteria: $('#st-reviews-tmpl-edit-criteria').html(),
		criteriaField: $('#st-reviews-tmpl-field').html(),
		noFields: $('#st-reviews-tmpl-no-fields').html(),
		criteriaScoreField: $('#st-reviews-tmpl-sc-field').html(),
		noScoreFields: $('#st-reviews-tmpl-sc-no-fields').html(),
		editCriteriaFrame: null,
		editFieldsList: null
	};

	/**
	 * Initialize Reviews
	 * 
	 */
	ST_REVIEWS.init = function() {

		$('.strw-create-criteria').each(function(){

			var nameField,
				criteriaData,
				criteriaObj,
				criteriaCollection,
				criteriaList;

			criteriaData = $(this).find('.strw-criteria-data').val();
			criteriaObj = JSON.parse( _.unescape( criteriaData ) );

			nameField = $(this).find('.strw-criteria-name');
			criteriaCollection = new ST_REVIEWS.Collections.Criteria( criteriaObj );
			criteriaList = new ST_REVIEWS.Views.CriteriaList( { collection: criteriaCollection } );

			$(this).append( criteriaList.render().$el );

			// add new criteria
			$(this).find('.strw-add-criteria').click( function(e){
				
				e.preventDefault();

				ST_REVIEWS.addCriteria(nameField, criteriaCollection);
			});

			nameField.keypress( function(e) {

				if( e.which == 13 ) {

					ST_REVIEWS.addCriteria(nameField, criteriaCollection);
					return false;
				}
			});

		});

	};

	/**
	 * Add Criteria
	 * 
	 */
	ST_REVIEWS.addCriteria = function( nameField, criteriaCollection ) {

		var criteria = new ST_REVIEWS.Models.Criteria({ name: $.trim( nameField.val() ) });

		if( ! criteria.isValid() ) {
			
			STFR.show_notification( criteria.validationError, 'warning' );
			nameField.focus();
			
			return;
		}

		var has_duplicate = criteriaCollection.some(function( _criteria ) { 
	    
	        return criteria.get('name').toUpperCase() === _criteria.get('name').toUpperCase();
	    });

		if( has_duplicate ) {
			
			STFR.show_notification( STRW_PAGE_l10n.dup_criteria, 'warning' );
			nameField.focus();
			
			return;
		}

		criteriaCollection.add( criteria );
		nameField.val('').focus();
	}	

	/**
	 * Initialize Criteria Fields
	 *
	 * @param Object context - jQuery object
	 */
	ST_REVIEWS.init_fields = function( context ) {

		if( _.isUndefined( context ) || _.isNull( context ) ) {
			return;
		}

		var fields = context.find('.strw-criteria-fields'),
			nameField,
			addFieldButton,
			fieldsData,
			fieldsObj,
			fieldsCollection,
			fieldsList;

		if( ! fields ) {
			return;
		}

		nameField = fields.find('.strw-field-name');
		addFieldButton = fields.find('.strw-add-field');
		fieldsData = fields.find('.strw-fields-data').val();
		fieldsObj = JSON.parse( _.unescape( fieldsData ) );

		fieldsCollection = new ST_REVIEWS.Collections.CriteriaFields( fieldsObj );
		fieldsList = new ST_REVIEWS.Views.CriteriaFieldsList( { collection: fieldsCollection } );

		ST_REVIEWS.editFieldsList = fieldsList;
		
		// render and add fields list	
		fields.append( fieldsList.render().$el );

		// make fields sortable
		fieldsList.$el.sortable({ placeholder: 'strw-field strw-field-placeholder' });

		// add new criteria
		addFieldButton.click( function(e){
			
			e.preventDefault();
			ST_REVIEWS.addCriteriaField(nameField, fieldsCollection);
		});

		nameField.keypress( function(e) {

			if( e.which == 13 ) {
				ST_REVIEWS.addCriteriaField(nameField, fieldsCollection);
				return false;
			}
		});
	}

	/**
	 * Destroy Criteria Fields
	 *
	 * @param Object context - jQuery object
	 */
	ST_REVIEWS.destroy_fields = function( context ) {

		if( _.isUndefined( context ) || _.isNull( context ) ) {
			return;
		}

		var nameField = context.find('.strw-field-name'),
			addFieldButton = context.find('.strw-add-field'),
			fieldsList = context.find('.strw-fields-list');

		nameField.off('keypress');
		addFieldButton.off('click');
		fieldsList.sortable('destroy');

		if( ! _.isNull( ST_REVIEWS.editFieldsList ) ) {
			
			ST_REVIEWS.editFieldsList.destroy();
			ST_REVIEWS.editFieldsList = null;
		}
	}

	/**
	 * Add Criteria Field
	 * 
	 */
	ST_REVIEWS.addCriteriaField = function( nameField, fieldsCollection ) {

		var field = new ST_REVIEWS.Models.CriteriaField({ name: $.trim( nameField.val() ) });

		if( ! field.isValid() ) {
			
			STFR.show_notification( field.validationError, 'warning' );
			nameField.focus();
			
			return;
		}

		var has_duplicate = fieldsCollection.some(function( _field ) { 
	    
	        return field.get('name').toUpperCase() === _field.get('name').toUpperCase();
	    });

		if( has_duplicate ) {
			
			STFR.show_notification( STRW_PAGE_l10n.dup_field, 'warning' );
			nameField.focus();
			
			return;
		}

		fieldsCollection.add( field );
		nameField.val('').focus();
	}

	/**
	 * Initialize Review Post
	 * 
	 * 
	 */
	ST_REVIEWS.init_review_post = function() {

		var container = $('.strw-review-score-wrap'),
			totalScore = container.find('.score'),
			selectCriteria = container.find('.strw-select-criteria'),
			fieldsData = null,
			fieldsObj = null,
			fieldsCollection = null,
			fieldsList = null;

		if( ! container.length ) {
			return;
		}

		ST_REVIEWS.defaults.totalScoreView = totalScore;

		fieldsCollection = ST_REVIEWS.set_review_fields( selectCriteria );
		fieldsList = new ST_REVIEWS.Views.CriteriaScoreList( { collection: fieldsCollection } );

		// render and add fields list	
		container.append( fieldsList.render().$el );
		
		// init slider inputs
		STFR.slider_input.add_to( fieldsList.$el );

		// handle select criteria
		selectCriteria.on('change', function(){

			STFR.slider_input.remove_from( fieldsList.$el );

			fieldsCollection = ST_REVIEWS.set_review_fields( selectCriteria );
			fieldsList.setCollection( fieldsCollection );
			
			STFR.slider_input.add_to( fieldsList.$el );
		});
	}

	/**
	 * Set Review Score Fields
	 *
	 * @return CriteriaScore Collection
	 */
	ST_REVIEWS.set_review_fields = function( selectCriteria ) {

		var fieldsData = selectCriteria.find('option:selected').attr('data-fields'),
			fieldsObj = null;

		if( fieldsData == 'null' ) {
			
			fieldsObj = [];

		} else {
			
			fieldsObj = JSON.parse( _.unescape( fieldsData ) );
		}

		return new ST_REVIEWS.Collections.CriteriaScore( fieldsObj );
	}

	/*---------------------------*
	 * Models
	 *---------------------------*/
	ST_REVIEWS.Models = {};

	/**
	 * Criteria Model
	 */
	ST_REVIEWS.Models.Criteria = Backbone.Model.extend({

		defaults: {
			name: 'Default Name',
			options: {}
		},

		validate: function(atts) {

			if( ! _.isString( atts.name ) ) {
				return 'Criteria name is not a string.';
			}

			if( $.trim( atts.name ).length < 2 ) {
				return STRW_PAGE_l10n.short_criteria;
			}
		}
	});

	/**
	 * Criteria Field Model
	 */
	ST_REVIEWS.Models.CriteriaField = Backbone.Model.extend({

		defaults: {
			name: 'Default Name'
		},

		validate: function(atts) {

			if( ! _.isString( atts.name ) ) {
				return 'Criteria name is not a string.';
			}

			if( $.trim( atts.name ).length < 2 ) {
				return STRW_PAGE_l10n.short_criteria_field;
			}		
		}
	});

	/**
	 * Criteria Score Model
	 */
	ST_REVIEWS.Models.CriteriaScore = Backbone.Model.extend({

		defaults: {
			uid: 'default_uid',
			name: 'Default Name',
			score: 0,
			minScore: 0.5,
			maxScore: 10,
			scoreStep: 0.5
		},
		
		initialize : function() {

			this.set('cid', this.cid );
			
			var uid = this.get('name').replace( /[ '"]/g, '_') + '_' + this.get('cid');
			
			this.set('uid', uid );

			if( this.get('score') == 0 && this.get('maxScore') > 0 ) {
				
				this.set('score', this.get('maxScore') / 2 );
			}

			if( this.get('score') > this.get('maxScore') ) {

				this.set('score', this.get('maxScore'));
			}
		},

		validate: function(atts) {

			if( ! _.isString( atts.name ) ) {
				return 'Criteria name is not a string.';
			}

			if( $.trim( atts.name ).length < 2 ) {
				return STRW_PAGE_l10n.short_criteria_field;
			}

			if( atts.score < 0 ) {
				return 'Score can not be negative.';
			}

			if( atts.maxScore <= 0 ) {
				return 'Maximum score should be greater than zero.';
			}
		}
	});

	/*---------------------------*
	 * Collections
	 *---------------------------*/
	ST_REVIEWS.Collections = {};

	/**
	 * Criteria Collection
	 * 
	 */
	ST_REVIEWS.Collections.Criteria = Backbone.Collection.extend({
		model: ST_REVIEWS.Models.Criteria
	});

	/**
	 * Criteria Fields Collection
	 * 
	 */
	ST_REVIEWS.Collections.CriteriaFields = Backbone.Collection.extend({
		model: ST_REVIEWS.Models.CriteriaField
	});

	/**
	 * Criteria Score Collection
	 * 
	 */
	ST_REVIEWS.Collections.CriteriaScore = Backbone.Collection.extend({
		model: ST_REVIEWS.Models.CriteriaScore
	});

	/*---------------------------*
	 * Views
	 *---------------------------*/
	ST_REVIEWS.Views = {};

	/**
	 * Criteria List View
	 * 
	 */
	ST_REVIEWS.Views.CriteriaList = Backbone.View.extend({
		
		tagName: 'ul',
		className: 'strw-criteria-list',

		initialize: function() {

			this.collection.on( 'add', this.addItem, this );
			this.collection.on( 'remove', this.setContentState, this );
		},

		render: function() {

			this.collection.each( this.addItem, this );
			this.setContentState();

			return this;
		},

		addItem: function(model) {
			
			var item = new ST_REVIEWS.Views.Criteria({ model: model, collection: this.collection }),
				element = item.render().$el;

			this.$el.append( element );
			element.animate({opacity: 1}, 200, 'easeOutQuad');

			this.setContentState();
		},

		setContentState: function() {

			if( this.collection.length == 0 ) {

				var placeholder = new ST_REVIEWS.Views.NoCriteria();
				this.$el.append( placeholder.render().$el );

			} else if( this.$el.find('.strw-empty-list') ) {
				
				this.$el.find('>.strw-empty-list').first().remove();
			}
		}
	});

	/**
	 * Criteria View
	 * 
	 */
	ST_REVIEWS.Views.Criteria = Backbone.View.extend({
		
		tagName: 'li',
		className: 'strw-criteria',
		template: _.template( ST_REVIEWS.templates.criteria ),

		events: {
			'click .strw-edit-icon': 'edit',
			'click .strw-delete-icon': 'delete',
		},

		initialize: function() {
			
			this.model.on( 'destroy', this.destroy, this );
			this.model.on( 'change', this.render, this );
			
		},

		render: function() {
			
			this.$el.html( this.template( this.model.toJSON() ) );
			
			return this;
		},

		edit: function(e) {
			
			e.preventDefault();

			if( ST_REVIEWS.defaults.editCriteriaFrame == null ) {
				
				// create edit frame
				ST_REVIEWS.defaults.editCriteriaFrame = new ST_REVIEWS.Views.EditCriteriaFrame();
				
				$('body').append( ST_REVIEWS.defaults.editCriteriaFrame.render().$el );
			}

			// open edit frame
			ST_REVIEWS.defaults.editCriteriaFrame.open( this.model );
		},

		delete: function(e) {
			
			e.preventDefault();
			this.model.destroy();
			this.model.off();
		},

		destroy: function() {

			this.$el.off();
			this.undelegateEvents();

			this.$el.animate({opacity: 0}, 200, 'easeOutQuad', function(){
				this.remove();
			});
		}

	});

	/**
	 * No Criteria
	 * 
	 * A view rendered when criteria list is empty.
	 * 
	 */
	ST_REVIEWS.Views.NoCriteria = Backbone.View.extend({
		
		tagName: 'li',
		className: 'strw-criteria strw-empty-list',
		template: _.template( ST_REVIEWS.templates.noCriteria ),

		render: function() {
			
			this.$el.html( this.template() );
			
			return this;
		}
	});

	/**
	 * Criteria Fields List View
	 *
	 * @extends CriteriaList
	 */
	ST_REVIEWS.Views.CriteriaFieldsList = ST_REVIEWS.Views.CriteriaList.extend({
		
		className: 'strw-fields-list',

		addItem: function(model) {
			
			var item = new ST_REVIEWS.Views.CriteriaField({ model: model }),
				element = item.render().$el;

			this.$el.append( element );
			element.animate({opacity: 1}, 200, 'easeOutQuad');

			this.setContentState();
		},

		setContentState: function() {

			if( this.collection.length == 0 ) {

				var placeholder = new ST_REVIEWS.Views.NoFields();
				this.$el.append( placeholder.render().$el );

			} else if( this.$el.find('.strw-empty-list') ) {
				
				this.$el.find('>.strw-empty-list').first().remove();
			}
		},

		destroy: function() {
			
			this.collection.off();
			this.undelegateEvents();

			while ( field = this.collection.first() ) {
				field.destroy();
				field.off();
			}
		}
	});

	/**
	 * Criteria Field View
	 *
	 * @extends Criteria
	 */
	ST_REVIEWS.Views.CriteriaField = ST_REVIEWS.Views.Criteria.extend({
		
		className: 'strw-field',
		template: _.template( ST_REVIEWS.templates.criteriaField )
	});

	/**
	 * No Criteria Fields
	 * 
	 * A view rendered when criteria list is empty.
	 * 
	 */
	ST_REVIEWS.Views.NoFields = Backbone.View.extend({
		
		tagName: 'li',
		className: 'strw-field strw-empty-list',
		template: _.template( ST_REVIEWS.templates.noFields ),

		render: function() {
			
			this.$el.html( this.template() );
			return this;
		}
	});

	/**
	 * Edit Criteria Frame View
	 * 
	 */
	ST_REVIEWS.Views.EditCriteriaFrame = Backbone.View.extend({

		tagName: 'div',
		className: 'strw-edit-frame-wrap',
		template: _.template( ST_REVIEWS.templates.editCriteria ),
		optionsForm: null,
		optionsWrap: null,
		loader: null,
		request: null,

		events: {
			'click .media-modal-backdrop': 'close',
			'click .media-modal-close': 'close',
			'click .button-save': 'save'
		},

		render: function() {
			
			this.$el.html( this.template() );
			
			this.optionsForm = this.$el.find('#strw-criteria-options-form');
			this.optionsWrap = this.$el.find('.strw-criteria-options');
			this.loader = this.$el.find('.loader');

			return this;
		},

		open: function( criteria ) {
			
			this.criteria = criteria;
			this.$el.show();
			this.loader.fadeIn(300);

			this.addContent( criteria );
		},

		close: function(e) {
			
			if(e) {
				e.preventDefault();
			}

			// unbind options events
			STFR.destroy_options( this.optionsWrap );
			ST_REVIEWS.destroy_fields( this.optionsWrap );

			// cancel AJAX request
			if( this.request ) {
				this.request.abort();
			}

			// hide frame and remove options
			this.loader.fadeOut(0);
			this.$el.hide();
			this.optionsWrap.empty();

			STFR.defaults.notification_view.css( 'z-index', 999 );
		},

		save: function(e) {
			
			var serialized_array = this.optionsForm.serializeArray(),
				options = {};

			_.each( serialized_array, function( item ) {
				
				if( ! _.isUndefined( options[item.name] ) ) {
					
					options[item.name].push( item.value );

				} else if( item.name.indexOf('[]') != -1 ) {

					options[item.name] = [ item.value ];
				
				} else {
					
					options[item.name] = item.value;
				}
				
			});

			// clean options object
			options['strw_criteria_fields'] = options['strw_criteria_field[]'];
			delete options['strw_criteria_field[]'];
			delete options['field-name'];

			// set criteria options
			this.criteria.set( 'options', options );

			this.close(e);
		},

		addContent: function(criteria) {

			this.request = $.ajax({
				context: this,
		        type: 'post',
		        url: ajaxurl,
		        data: { 
		        	action: 'st_reviews_edit_criteria',
		        	options: criteria.get('options'),
		        	nonce: STRW_PAGE_l10n.nonce
		        },
		        success: function(data) {

		        	// hide loader
		        	this.loader.fadeOut(300);

		        	// add generated options to container
		        	this.optionsWrap.html( data );

		        	// init options
		        	STFR.init_options( this.optionsWrap );
		        	ST_REVIEWS.init_fields( this.optionsWrap );

		        	STFR.defaults.notification_view.css( 'z-index', 999999 );
				}
			});
		}

	});
	
	/**
	 * Criteria Score List View
	 *
	 * @extends CriteriaList
	 */
	ST_REVIEWS.Views.CriteriaScoreList = ST_REVIEWS.Views.CriteriaList.extend({
		
		className: 'strw-score-fields-list',

		addItem: function(model) {
			
			var item = new ST_REVIEWS.Views.CriteriaScoreField({ model: model }),
				element = item.render().$el;

			model.on('change:score', this.updateTotalScore, this);

			this.$el.append( element );
			element.animate({opacity: 1}, 200, 'easeOutQuad');

			this.setContentState();
		},

		setContentState: function() {

			if( this.collection.length == 0 ) {

				var placeholder = new ST_REVIEWS.Views.NoScoreFields();
				this.$el.append( placeholder.render().$el );

			} else if( this.$el.find('.strw-empty-list') ) {
				
				this.$el.find('>.strw-empty-list').first().remove();
			}

			this.updateTotalScore();
		},

		destroy: function() {
			
			this.collection.off();
			this.undelegateEvents();

			while ( field = this.collection.first() ) {
				field.destroy();
				field.off();
			}
		},

		setCollection: function( collection ) {

			this.collection.off('remove');

			while ( field = this.collection.first() ) {
				field.destroy();
				field.off();
			}

			this.collection = collection;
			this.render();
		},

		updateTotalScore: function() {

			var totalScore = '0.0',
				sum = 0;

			this.collection.each( function( model ) {

				sum += parseFloat( model.get('score') );
			});

			if( this.collection.length > 0 ) {
				
				totalScore = ( sum / this.collection.length ).toFixed(1);
			}

			ST_REVIEWS.defaults.totalScoreView.html( totalScore );
		}
	});

	/**
	 * Criteria Score Field View
	 * 
	 */
	ST_REVIEWS.Views.CriteriaScoreField = Backbone.View.extend({
		
		tagName: 'li',
		className: 'strw-score-field',
		template: _.template( ST_REVIEWS.templates.criteriaScoreField ),
		
		initialize: function() {
			
			this.model.on( 'destroy', this.destroy, this );
			this.model.on( 'change', this.renderStucture, this );
			
		},

		render: function() {
			
			this.$el.html( this.template( this.model.toJSON() ) );
			
			if( _.isUndefined( this.slider ) ) {
				
				this.slider = this.$el.find('.stfr-ui-slider');
				this.slider.on( 'slide', jQuery.proxy( this, 'sliderChange' ) );
				
				this.renderStucture();
			}

			return this;
		},

		renderStucture: function() {

			var template = $('#st-reviews-tmpl-sc-data').html(),
				structure = _.template( template, this.model.toJSON() ),
				oldStructure = this.$el.find('.strw-field-data-wrap');

			if( oldStructure ) {
				oldStructure.remove();
			}

			this.$el.append( structure );
		},

		destroy: function() {

			if( ! _.isUndefined( this.slider ) ) {
				this.slider.off();
			}

			this.$el.off();
			this.undelegateEvents();
			this.remove();
		},

		sliderChange: function( e, ui ) {
			
			this.model.set('score', ui.value );
		}
	});

	/**
	 * No Score Fields
	 * 
	 * A view rendered when criteria score fields list is empty.
	 * 
	 */
	ST_REVIEWS.Views.NoScoreFields = Backbone.View.extend({
		
		tagName: 'li',
		className: 'strw-field strw-empty-list',
		template: _.template( ST_REVIEWS.templates.noScoreFields ),

		render: function() {
			
			this.$el.html( this.template() );
			return this;
		}
	});

	ST_REVIEWS.init();
	ST_REVIEWS.init_review_post();
});