<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'SixthType_Meta_Box' ) ) :

/**
 * SixthType Meta Box
 *
 * Provides methods for adding meta boxes to certain post types,
 * saving meta box options and displaying meta box options in WordPress admin panel.
 *
 * @version 	1.0
 * @since 		1.0
 * @package 	SixthType-Framework/Includes
 * @author 		SixthType <hello@sixthtype.com>
 */
class SixthType_Meta_Box {

	/**
	 * Singleton instance
	 *
	 * @static
	 * @access 	private
	 * @since 	1.0
	 * @var 	object - SixthType_Meta_Box instance
	 */
	private static $instance = null;

	/**
	 * Holds meta boxes
	 *
	 * @access 	private
	 * @since  	1.0
	 * @var 	array
	 */
	private $meta_boxes;

	/**
	 * Holds post types meta boxes
	 *
	 * @access 	private
	 * @since  	1.0
	 * @var 	array
	 */
	private $post_meta_boxes;

	/**
	 * Holds options view object
	 * 
	 * @access   private
	 * @since    1.0
	 * @var      SixthType_Options_View
	 */
	private $options_view;

	/**
	 * Constructor
	 *
	 * @access 	public
	 * @since 	1.0
	 */
	private function __construct() {

		$this->set_options_view();
		$this->post_meta_boxes = array();

		add_action( 'init', array( &$this, 'init' ), 300 );	
		add_action( 'save_post', array( &$this, 'save' ), 10, 2 );
		add_action( 'add_meta_boxes', array( &$this, 'add_meta_boxes' ) );
	}

	/**
	 * Get Instance
	 *
	 * Use this method to obtain SixthType_Meta_Box instance
	 *
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @return 	SixthType_Meta_Box - instance object
	 */
	public static function get_instance() {

		if ( ! ( self::$instance instanceof SixthType_Meta_Box ) ) {
      		
      		self::$instance = new SixthType_Meta_Box();
		}
    	
    	return self::$instance;
	}

	/**
	 * Set Options View
	 *
	 * @access  public
	 * @since   1.0
	 * @param   SixthType_Options_View $options_view - options view object to use
	 */
	public function set_options_view( $options_view = null ) {

		if( ! isset( $options_view ) || ! $options_view instanceof SixthType_Options_View ) {

			$this->options_view = SixthType_Options_View::get_instance();
		
		} else {
			
			$this->options_view = $options_view;
		}
	}

	/**
	 * Initialize
	 *
	 * @access  public
	 * @since   1.0
	 */
	public function init(){
		
		$meta_boxes = array();

		if( function_exists('stfr_get_meta_boxes') && class_exists( 'SixthType_Framework' ) ) {

			$meta_boxes = stfr_get_meta_boxes();
		}

		$this->post_meta_boxes = apply_filters( 'stfr_setup_posts_meta_boxes', $this->post_meta_boxes );
		$this->meta_boxes = apply_filters( 'stfr_setup_meta_boxes', $meta_boxes );;
	}

	/**
	 * Add Meta Boxes
	 *
	 * Used as callback function for 'add_meta_boxes' hook
	 * to add all registered via add_post_meta_box() method metaboxes.
	 *
	 * @access 	public
	 * @since  	1.0
	 */
	public function add_meta_boxes() {

		foreach ( $this->post_meta_boxes as $element ) {
			
			foreach ( $element['meta_boxes'] as $meta_box_name ) {
				
				// if there is no meta box with this name continue to next one
				if( ! array_key_exists( $meta_box_name , $this->meta_boxes ) ) {
					continue;
				}

				$meta_box = $this->meta_boxes[ $meta_box_name ];

				// add meta box to post
				add_meta_box(
					$meta_box['id'],
					$meta_box['title'],
					array( &$this, 'display' ),
					$element['post_type'],
					$meta_box['context'],
					$meta_box['priority'],
					$meta_box
				);
			}
		}
	
	}

	/**
	 * Add Post Meta Box
	 *
	 * Used to register post meta boxes
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	string 	$post_type
	 * @param 	array $meta_boxes - meta box name or array with names
	 */
	public function add_post_meta_box( $post_type, $meta_boxes ) {

		// verify that post type exists and meta boxes are not empty
		if( empty( $post_type ) || ! post_type_exists( $post_type ) ||
			empty( $meta_boxes ) || ! is_array( $meta_boxes ) ) {
			
			return;
		}

		// add element to $post_meta_boxes
		$this->post_meta_boxes[] = array( 
			'post_type' => $post_type,
			'meta_boxes' => $meta_boxes
		);
	}

	/**
	 * Display Meta Box
	 *
	 * Used as callback function to generate meta box content.
	 *
	 * @access 	public
	 * @since 	1.0
	 * @param 	WP_Post	$post
	 * @param  	array 	$options - callback arguments
	 * @uses   	SixthType_Options_View object to display meta box options
	 */
	public function display( $post, $arguments = array() ) {

		// get meta box array from passed callback arguments
		$meta_box = $arguments['args'];

		// check if meta box will use custom meta box view using action hook
		if( ! empty( $meta_box['display_action'] ) ) {

			// call display action
			do_action( $meta_box['display_action'], $meta_box );

		} else {

			// display options using SixthType_Options_View object
			echo '<div class="stfr-options-view stfr-meta-box-view">';
			
			$options = call_user_func( $meta_box['options'] );
			$this->options_view->generate_meta_box_options( $options, $post->ID );
			
			echo '</div>';
		}

		// add meta box nonce field
		wp_nonce_field( $meta_box['id'], $meta_box['id'] );
	}

	/**
	 * Save
	 *
	 * Saves meta box data
	 *
	 * @access 	public
	 * @since 	1.0
	 * @param  	integer $post_id
	 * @param  	WP_Post $post
	 */
	public function save( $post_id, $post ) {

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) { 
	      	return;
		}

		if ( ! current_user_can( 'edit_pages', $post->ID ) ) {
			return $post->ID;
		}

		// find which meta box we need to save
		// and verify each meta box nonce field
		foreach ( $this->post_meta_boxes as $element ) {

			foreach ( $element['meta_boxes'] as $meta_box_name ) {

				// if there is no meta box with this name continue to next one
				if( ! array_key_exists( $meta_box_name , $this->meta_boxes ) ) {
					continue;
				}

				$meta_box = $this->meta_boxes[ $meta_box_name ];

				if( ! isset( $_POST[ $meta_box['id'] ] ) ){
					continue;
				}

				if ( ! wp_verify_nonce( $_POST[ $meta_box['id'] ], $meta_box['id'] ) ) {
					continue;
			 	}

			 	// check if meta box post type and current post type match
			  	if ( isset( $_POST['post_type'] ) && $_POST['post_type'] === $element['post_type'] ) {

			  		// if meta box have save action, call it
			  		if( isset( $meta_box['save_action'] ) ) {

			  			do_action( $meta_box['save_action'], $post_id, $meta_box );

			  			// continue to next meta box
			  			continue;
					}

			  		$this->save_options( $post_id, $meta_box );
				}
			}
		}

	}

	/**
	 * Save Options
	 * 
	 * @access 	private
	 * @since  	1.0
	 * @param  	integer $post_id
	 * @param   array $meta_box
	 * @uses    SixthType_Validator - to validate input data
	 */
	private function save_options( $post_id, $meta_box ) {

		if( ! is_numeric( $post_id ) || $post_id < 0 ) {
			return;
		}

		if( empty( $meta_box ) ) {
			return;
		}

		$post_id = intval( $post_id );
		$meta_box_options = call_user_func( $meta_box['options'] );

		foreach ( $meta_box_options as $meta_box_option ) {
  			
  			if( ! isset( $_POST[ $meta_box_option['id'] ] ) ) {
  				
  				// deal with checkboxes
  				if( $meta_box_option['type'] == 'checkbox' ) {

  					update_post_meta( $post_id, $meta_box_option['id'], 'off' );
				}

  				continue;
  			}

  			$data = SixthType_Validator::validate_option( $_POST[ $meta_box_option['id'] ], $meta_box_option );
  			
  			if( is_wp_error( $data ) ) {
  				
  				wp_die( $data );
  			
  			} else {

  				update_post_meta( $post_id, $meta_box_option['id'], $data );
  			}
  		}

	}

	/**
	 * Prevent object clone
	 *
	 * @access 	private
	 * @since 	1.0
	 */
	private function __clone() {}
}

endif; // end class exists check

?>