<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'SixthType_Options_View' ) ) :

/**
 * SixthType Options View
 *
 * Generates meta box and page options view
 * 
 * @version 	1.0
 * @since 		1.0
 * @package 	SixthType-Framework/Includes
 * @author 		SixthType <hello@sixthtype.com>
 */
class SixthType_Options_View {

	/**
	 * Singleton instance
	 *
	 * @static
	 * @access 	private
	 * @since 	1.0
	 * @var 	object - SixthType_Options_View instance
	 */
	private static $instance = null;

	/**
	 * Determine if we are working with settings page option
	 *
	 * @access 	private
	 * @since  	1.0
	 * @var 	boolean
	 */
	private $is_page_option;

	/**
	 * Holds generated menu
	 * 
	 * @access 	private
	 * @since 	1.0
	 * @var 	string
	 */
	private $menu;
	
	/**
	 * Used to close menu items group
	 *
	 * @access 	private
	 * @since 	1.0
	 * @var 	integer
	 */
	private $menu_counter;

	/**
	 * Used to cache fonts icons list
	 *
	 * @access 	private
	 * @since 	1.0
	 * @var 	array
	 */
	private $fonts_icons_cache;

	/**
	 * Constructor
	 * 
	 * @access 	private
	 * @since 	1.0
	 */
	private function __construct() {
		
		$this->is_page_option = false;
		$this->menu = '';
		$this->menu_counter = 0;
		$this->font_icons_cache = array();

		// add action hooks for default options view actions
		
		// heading option
		add_action( 'stfr_show_heading_option', array( &$this, 'heading' ), 10, 3 );

		// text option
		add_action( 'stfr_show_text_option', array( &$this, 'text' ), 10, 3 );

		// textarea option
		add_action( 'stfr_show_textarea_option', array( &$this, 'textarea' ), 10, 3 );

		// checkbox option
		add_action( 'stfr_show_checkbox_option', array( &$this, 'checkbox' ), 10, 3 );

		// radio option
		add_action( 'stfr_show_radio_option', array( &$this, 'radio' ), 10, 3 );

		// select option
		add_action( 'stfr_show_select_option', array( &$this, 'select' ), 10, 3 );

		// select image option 
		add_action( 'stfr_show_select_image_option_option', array( &$this, 'select_image_option' ), 10, 3 );

		// slider input option
		add_action( 'stfr_show_slider_input_option', array( &$this, 'slider_input' ), 10, 3 );

		// color picker option
		add_action( 'stfr_show_color_picker_option', array( &$this, 'color_picker' ), 10, 3 );

		// switch toggle
		add_action( 'stfr_show_switch_toggle_option', array( &$this, 'switch_toggle' ), 10, 3 );

		// file option
		add_action( 'stfr_show_file_option', array( &$this, 'file' ), 10, 3 );

		// gallery option
		add_action( 'stfr_show_gallery_option', array( &$this, 'gallery' ), 10, 3 );

		// editor option
		add_action( 'stfr_show_editor_option', array( &$this, 'editor' ), 10, 3 );

		// checkboxes option
		add_action( 'stfr_show_checkboxes_option', array( &$this, 'checkboxes' ), 10, 3 );

		// font option
		add_action( 'stfr_show_font_option', array( &$this, 'font' ), 10, 3 );
		
		// icon picker option
		add_action( 'stfr_show_icon_picker_option', array( &$this, 'icon_picker' ), 10, 3 );	
	}

	/**
	 * Get Instance
	 *
	 * Use this method to obtain SixthType_Options_View instance
	 *
	 * @static
	 * @since  	1.0
	 * @access 	public
	 * @return 	SixthType_Options_View 	- instance object
	 */
	public static function get_instance() {

		if ( ! ( self::$instance instanceof SixthType_Options_View ) ) {
      		
      		self::$instance = new SixthType_Options_View();
		}
    	
    	return self::$instance;
	}

	/**
	 * Prevent object clone
	 *
	 * @since 	1.0
	 * @access 	private
	 */
	private function __clone() {}

	/**
	 * Call Option Hook
	 *
	 * @access  public
	 * @since 	1.0
	 * @param  	array  $option
	 * @param  	mixed  $value - option value
	 */
	public function call_option_hook( $option = array(), $value ) {

		if( empty( $option ) || ! isset( $value ) ) {
			return;
		}

		$option_type = str_replace( array( ' ', '-' ), '_', $option['type'] );
		$action = 'stfr_show_' . $option_type . '_option';

		do_action( $action, $option, $this, $value );

		// add options separator line
		echo '<div class="line"></div>';
	}

	/**
	 * Generate Options Page
	 *
	 * Generates page options view and menu
	 * 
	 * @access 	public
	 * @since 	1.0
	 * @param   $options - array with page options
	 * @param   $saved_options - array with saved options [key] => [value]
	 */
	public function generate_page_options( $options, $saved_options ) {
		
		if( ! is_array( $options ) || ! is_array( $saved_options ) ) {

			return;
		}

		// reset menu counter
		$this->menu_counter = 0;
		$this->is_page_option = true;
		
		// loop thru options and call optons display callback functions
		foreach ( $options as $option ) {
			
			// set option name value
			if( ! isset( $option['name'] ) ) {
				$option['name'] = $option['id'];
			}

			$option_saved_value = null;

			if( isset( $saved_options[ $option['id'] ] ) ) {
				
				$option_saved_value = $saved_options[ $option['id'] ];
			}

			// add menu item
			if( $option['type'] == 'heading' ) {
					
				$this->add_menu_item( '<li><a id="' . esc_attr( $option['id'] ) . '"
											 title="' . esc_attr( $option['title'] ) . '"
											 href="#' . esc_attr( $option['id'] ) . '">'
											 . esc_html( $option['title'] ) . 
										'</a></li>');

			}

			// if option is empty get its default value
			if( ! isset( $option_saved_value ) || empty( $option_saved_value ) ) {
				
				$option_saved_value = $option['default'];
			}

			// call option hook to display the option
			$this->call_option_hook( $option, $option_saved_value );

			$this->menu_counter += 1;
		}

		$this->is_page_option = false;
	}

	/**
	 * Generate Meta Box Options
	 * 
	 * @access 	public
	 * @since 	1.0
	 * @param  	array   $options - options array
	 * @param  	integer $post_id - meta box post ID
	 */
	public function generate_meta_box_options( $options = array(), $post_id = 0 ) {

		if( ! is_array( $options ) || $post_id == 0 ) {
			return;
		}

		// loop thru options and call optons display callback functions
		foreach ( $options as $option ) {
			
			// set option name value
			if( ! isset( $option['name'] ) ) {
				$option['name'] = $option['id'];
			}

			$option_saved_value = get_post_meta( $post_id, $option['id'], true );
			
			// if option is empty get its default value
			if( empty( $option_saved_value ) ) {
				$option_saved_value = $option['default'];
			}

			// call option hook to display the option
			$this->call_option_hook( $option, $option_saved_value );
		}
	}

	/**
	 * Generate Options
	 *
	 * Generate options from array with options and
	 * array with saved options values
	 *
	 * @access  public
	 * @since  	1.0
	 * @param  	array  $options 
	 * @param  	array  $saved_values
	 */
	public function generate_options( $options = array(), $saved_values = array() ) {

		if( ! is_array( $options ) || ! is_array( $saved_values ) ) {
			return;
		}

		foreach( $options as $option ) {

			$option_value = null;

			// set option name value 
			if( ! isset( $option['name'] ) ) {
				$option['name'] = $option['id'];
			}

			if( array_key_exists( $option['id'], $saved_values ) ) {
				$option_value = $saved_values[ $option['id'] ];
			}

			if( ! isset( $option_value ) || empty( $option_value ) ) {
				$option_value = $option['default'];
			}

			$this->call_option_hook( $option, $option_value );

		}
	}

	/**
	 * Is Page Option
	 *
	 * Returns true if we are working with page options
	 * 
	 * @return boolean
	 */
	public function is_page_option() {
		
		return $this->is_page_option;
	}

	/**
	 * Add Menu Item
	 * 
	 * @param string $item - list element
	 */
	public function add_menu_item( $item ) {

		if( ! isset( $item ) || ! is_string( $item ) || trim( $item ) == '' ) {
			return;
		}

		$this->menu .= $item;
	}

	/**
	 * Get Menu
	 *
	 * @access 	public
	 * @since 	1.0
	 * @return 	string - generated menu
	 */
	public function get_menu() {
		
		return $this->menu;
	}

	/**
	 * Get Menu Counter
	 *
	 * @access 	public
	 * @since 	1.0
	 * @return 	integer - menu counter value
	 */
	public function get_menu_counter() {
	
		return $this->menu_counter;
	}

	/**
	 * Get Fonts Icons List
	 *
	 * @access 	public
	 * @since 	1.0
	 * @return 	array/WP_Error
	 */
	public function get_fonts_icons() {

		if( ! empty( $this->fonts_icons_cache ) ) {

			return $this->fonts_icons_cache;
		}

		$icon_font = apply_filters( 'stfr_icon_font', array() );
		$icons_list = SixthType_Utilities::get_font_css_icons( $icon_font['css_file'], $icon_font['prefix'] );
		
		if( ! is_wp_error( $icons_list ) ) {
			
			$this->fonts_icons_cache = $icons_list;
		}

		return $icons_list;
	}

	/**
	 * Display Option Title
	 *
	 * Outputs passed option name and description text
	 * 
	 * @access 	public
	 * @since 	1.0
	 * @param   array $option
	 */
	public function display_option_title( $option = array() ) {

		if( empty( $option ) ) {
			return;
		}

		if( isset( $option['title'] ) && ! empty( $option['title'] ) ) {
			echo '<h3>' . esc_html( $option['title'] ) . '</h3>';
		}

		if( isset( $option['desc'] ) && ! empty( $option['desc'] ) ) {
			echo '<p class="description">' . esc_html( $option['desc'] ) . '</p>';
		}
	}

	// ----------------------- Options Callbacks ----------------------- //
	
	/**
	 * Heading
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function heading( $option, $option_view, $option_value ) {

		// if we are displaing page options and we have added at least 1 menu group
		// check do we need to close menu options group
		if( $option_view->is_page_option() && $option_view->get_menu_counter() >= 1 ) {
			echo '</div>';
		}

		// if we are displaing page option add menu group
		if( $option_view->is_page_option() ) {
			echo '<div class="hidden group" id="' . esc_attr( $option['id'] ) . '">';
		}

		if( isset( $option['title'] ) && ! empty( $option['title'] ) ) {
			echo '<h2>' . esc_html( $option['title'] ) . '</h2>';
		}

		if( isset( $option['desc'] ) && ! empty( $option['desc'] ) ) {
			echo '<p class="description">' . esc_html( $option['desc'] ) . '</p>';
		}
	}

	/**
	 * Text
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function text( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		// output text input
		echo '<input class="' . esc_attr( $option['class'] ) . '" 
					 name="' . esc_attr( $option['name'] ) . '" 
					 type="text" 
					 value="' . esc_attr( $option_value ) . '" 
				/>';
	}

	/**
	 * Textarea
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function textarea( $option, $option_view, $option_value ) {

		$rows = 7;
		
		if( isset( $option['rows'] ) ) {

			$rows = intval( $option['rows'] );
		}

		// display option title
		$option_view->display_option_title( $option );

		echo '<textarea  rows="' . esc_attr( $rows ) . '" 
						 cols="25" 
						 class="' . esc_attr( $option['class'] ) . '" 
						 name="' . esc_attr( $option['name'] ) . '" >' . 
						 esc_textarea( $option_value ) . 
				'</textarea>';
	}

	/**
	 * Checkbox
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function checkbox( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		$checked = '';
		
		if( ! empty( $option_value ) && $option_value == 'on' ) {
			$checked = ' checked=checked';
		}

		echo '<input type="checkbox" 
					 id="' . esc_attr( $option['id'] ) . '" 
					 name="' . esc_attr( $option['name'] ) . '" ' . $checked . ' />';
		
		echo '<label for="' . esc_attr( $option['id'] ) . '" 
					class="' . esc_attr( $option['class'] ) . '" 
				>' . esc_html( $option['options']['label'] ) . '</label>';
	}

	/**
	 * Radio
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function radio( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		$counter = 0;
		
		foreach ( $option['options'] as $radio ) {
			
			$checked = '';

			$radio_id = esc_attr( $option['id']  ) . '_' . $counter;

			if( $option_value == $radio['value'] ) {
				$checked = ' checked=checked';
			}

			echo '<input type="radio" 
						 id="' . $radio_id . '" 
						 class="' . $option['class'] . '" 
						 name="' . esc_attr( $option['name'] ) . '" 
						 value="' . esc_attr( $radio['value'] ) . '"' . $checked . '>';

			echo '<label for="' . $radio_id . '" class="stfr-radio-label">' . esc_html( $radio['label'] ) . '</label>';

			$counter += 1;
		}
	}

	/**
	 * Select
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function select( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		echo '<select name="' . esc_attr( $option['name'] ) . '" 
					  class="' . esc_attr( $option['class'] ) . '">';

		foreach ( $option['options'] as $item ) {
			
			$selected = '';

			if( $option_value == $item['value'] ) {
				$selected = ' selected=selected';
			}

			echo '<option value="' . esc_attr( $item['value'] ) . '" ' . $selected . '>' . esc_html( $item['label'] ) . '</option>';
		}

		echo '</select>';
	}

	/**
	 * Select Image Option
	 *
	 * Displays a list with options represented as 
	 * images where user can click on image to select option.
	 * For example sidebar positions or list with
	 * background patterns.
	 * 
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function select_image_option( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		echo '<div class="' . esc_attr( $option['class'] ) . '" data-target-id="' . esc_attr( $option['id'] ) . '">';
		echo '<ul>';
		
			foreach ( $option['options'] as $value => $image ) {
				
				$checked = '';

				if( $value == $option_value ){
					$checked = 'selected';
				}

				echo '<li class="' . $checked . '">
						<a href="#" data-sel-val="' . esc_attr( $value ) . '" title="' . esc_attr( $value ) . '">
							<img src="' . esc_attr( $image ) . '" alt="' . esc_attr( $value ) . '" width="120" height="80" />
						</a>
					  </li>';

			}

		echo '</ul>
				<input type="hidden" 
						id="' . esc_attr( $option['id'] ) . '" 
						name="' . esc_attr( $option['id'] ) . '" 
						value="' . esc_attr( $option_value ) . '" 
				/>';

		echo '</div>';
	}

	/**
	 * Slider Input
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function slider_input( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		echo '<input id="' . esc_attr( $option['id'] ) . '" 
					 class="' . esc_attr( $option['class'] ) . '" 
					 name="' . esc_attr($option['id']) . '" 
					 type="text" 
					 value="' . esc_attr( $option_value ) . '" 
				/>';

		echo '<div 	data-value="' . esc_attr( $option_value ) . '" 
					data-field="' . esc_attr( $option['id'] ) . '" 
					data-min="' . esc_attr( $option['options']['min'] ) . '" 
					data-max="' . esc_attr( $option['options']['max'] ) . '" 
					data-step="' . esc_attr( $option['options']['step'] ) . '" 
					class="stfr-ui-slider" >';

			echo '<span class="ui-slider-handle"></span>';

		echo '</div>';
		
		echo '<div class="stfr-clear"></div>';
	}

	/**
	 * Color Picker
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function color_picker( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		echo '<div 	id="stfr-color-box-' . esc_attr( $option['id'] ) . '" 
					class="stfr-color-picker-preview" 
					style="background-color:#' . esc_attr( $option_value ) . ';">
				</div>';

		echo '<input data-color-box="stfr-color-box-' . esc_attr( $option['id'] ) . '" 
					 type="text" 
					 maxlength="6" 
					 size="6" 
					 id="' . esc_attr( $option['id'] ) . '" 
					 name="' . esc_attr( $option['id'] ) . '" 
					 class="'. esc_attr( $option['class'] ) .'" 
					 value="' . esc_attr( $option_value ) . '" 
				/>';

		echo '<div class="stfr-clear"></div>';
	}

	/**
	 * Switch Toggle
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function switch_toggle( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		echo '<div 	data-val-id="' . esc_attr( $option['id'] ) . '" 
					class="'. esc_attr( $option['class'] ) . ' state-' . esc_attr( $option_value ) . '">';

			echo '<div class="stfr-switch-overlay"></div>';
			echo '<div class="stfr-active-bg"></div>';

			echo '<input type="hidden" 
						 id="' . esc_attr( $option['id'] ) . '" 
						 name="' . esc_attr( $option['id'] ) . '" 
						 value="' . esc_attr( $option_value ) . '" />';
		
		echo '</div>';
	}

	/**
	 * File
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function file( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		$url = '';
		
		if( isset( $option_value['url'] ) ) {
			$url = $option_value['url'];
		}

		echo '<input id="' . esc_attr( $option['id'] ) . '-url" 
					 class="' . esc_attr( $option['class'] ) . '" 
					 type="text"  
					 value="' . esc_attr( $url ) . '" 
				/>';

		echo '<input type="hidden" 
					id="' . esc_attr( $option['id'] ) . '" 
					name="' . esc_attr( $option['id'] ) . '"
					value="' . esc_attr( json_encode( $option_value ) ) . '"
				/>';

		echo '<span class="button upload stfr-file-button"
					data-type="' . esc_attr( $option['options']['type'] ) . '"  
					data-title="' . esc_attr( $option['title'] ) . '" >'
				. esc_html( $option['options']['button_title'] )
				. '</span>';
		
		// if its an image file type display image preview
		if( $option['options']['type'] == 'image' ) {
			
			if( ! isset( $url ) || strlen( trim( $url ) ) == 0 ) {

				$url = plugin_dir_url( __FILE__ ) . '../admin/images/default-upload-image.jpg';
			}

			echo '<div>
					<img class="stfr-image-preview"
						 id="' . esc_attr( $option['id'] ) . '_preview" 
						 src="'. esc_url( $url ) .'" 
						 alt="' . __( 'Selected Image Preview', 'sixthtype-framework' ) . '"
					/>
				</div>';
		}

	}

	/**
	 * Gallery
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function gallery( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		$attachments = array();
		$slides_ids = array();
		$json_slides_ids = array();

		$query_args = array( 
			'post_type' => 'attachment',
			'post_mime_type' =>'image',
			'post_status' => 'inherit',
			'posts_per_page' => -1,
			'orderby' => 'post__in',
			'post__in' => $option_value
		);

		// execute WP_Query and get only attachments that are available in 
		// media library.
		
		if( ! empty( $option_value ) ) {
			
			$attachments = new WP_Query( $query_args );
		
			foreach( $attachments->posts as $slide ) {
				
				$slides_ids[] = $slide->ID;
				$json_slides_ids[] = $slide->ID;
			}

			wp_reset_query();
		}

		echo '<span class="button upload stfr-gallery-button"   
					data-title="' . esc_attr( $option['title'] ) . '" >'
				. esc_html( $option['options']['button_title'] )
				. '</span>';

		echo '<input type="hidden" 
					id="' . esc_attr( $option['id'] ) . '" 
					name="' . esc_attr( $option['id'] ) . '"
					value="' . esc_attr( json_encode( $json_slides_ids ) ) . '"
				/>';

		echo '<ul id="' . esc_attr( $option['id'] ) . '_slides" 
				  class="' . esc_attr( $option['class'] ) . '">';
		
		echo '<li class="template">
					<img src="' . plugin_dir_url( __FILE__ ) . '../admin/images/gallery-slide-default.png"
						 width="75" 
						 alt="Template"/>
				</li>';

		// output saved attachments
		if( ! empty( $slides_ids ) ) {
		
			foreach ( $slides_ids as $id ) {
				
				$image = wp_get_attachment_image_src( $id, 'thumbnail' );
				echo '<li><img src="' . esc_attr( $image[0] ) . '" width="75" alt="Slide"/></li>';
			}
		
		}else{
			
			echo '<li class="no-slides">
					' . __('No images selected.', 'sixthtype-framework') . '
					</li>';
		}

		echo '</ul>';
	}

	/**
	 * Editor
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function editor( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );

		$settings = array(
			'textarea_name' => $option['id'],
			'textarea_rows' => 7,
			'media_buttons' => false,
			'tinymce' => false
		);
			
		if( ! empty( $option['options'] ) ) {
			$settings = array_merge( $settings, $option['options'] );
		}

		echo '<div class="' . esc_attr( $option['class'] ) . '">';
		
		wp_editor( $option_value, $option['id'] . '-editor', $settings );
		
		echo '</div>';
	}

	/**
	 * Checkboxes
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function checkboxes( $option, $option_view, $option_value ) {

		if( ! is_array( $option_value ) ) {
			return;
		}

		// display option title
		$option_view->display_option_title( $option );

		$counter = 0;

		foreach ( $option['options'] as $checkbox ) {
			
			$checked = '';
			
			$cbx_id = esc_attr( $option['id']  ) . '_' . $counter;

			if( in_array( $checkbox['value'], $option_value ) ) {
				$checked = ' checked=checked';
			}

			echo '<input type="checkbox" 
						 id="' . esc_attr( $cbx_id ) . '" 
						 name="' . esc_attr( $option['name'] ) . '[]" ' . $checked . '
						 value="'. esc_attr( $checkbox['value'] ) .'" />';
			
			echo '<label for="' . esc_attr( $cbx_id ) . '" 
						class="' . esc_attr( $option['class'] ) . '" 
					>' . esc_html( $checkbox['label'] ) . '</label>';

			$counter += 1;

		}
	}

	/**
	 * Font
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function font( $option, $option_view, $option_value ) {

		if( ! is_array( $option_value ) ) {
			return;
		}

		// display option title
		$option_view->display_option_title( $option );

		$fonts = SixthType_Utilities::get_fonts();
		$json_value = json_encode( $option_value );

		echo '<div class="' . esc_attr( $option['class'] ) . '">';
		
		echo '<input type="hidden" class="font-data" name="' . esc_attr( $option['name'] ) . '" value="' . esc_attr( $json_value ) . '"/>';

		echo '<select class="fonts-list stfr-select stfr-left">';

		foreach( $fonts as $font_group => $fonts ) {
			
			if( empty( $fonts ) ) {

				continue;
			}

			$group_name = str_replace( '-', ' ', $font_group );

			echo '<optgroup label="' . esc_attr( $group_name ) . '">';

			foreach( $fonts as $font ) {
				
				$selected = '';

				if( $option_value['family'] == $font['family'] ) {
					$selected = ' selected="selected"';
				}

				$font_data = json_encode( $font );

				echo '<option data-font="' . esc_attr( $font_data ) . '"
							  value="' . esc_attr( $font['family'] ) . '"
							  ' . $selected . '>' . esc_html( $font['family'] ) .
					  '</option>';

			}

			echo '</optgroup>';
			
		}

		echo '</select>';

		$font_style = 'style="font-family: ' . esc_attr( $option_value['family'] ) . '"';
		$font_face = SixthType_Utilities::get_font_face( $option_value );
		$inline_font = '#';

		if( $font_face != '' ) {
			
			$inline_font = 'href="' . esc_url( $font_face ) . '"/>';
		}

		// font preview
		echo 	'<div class="stfr-font-preview stfr-left" ' . $font_style . '>' . 
					'<link class="font-face" itemprop="font-preview" type="text/css" href="' . esc_url( $inline_font ) . '"/>' .
					__( 'Font Preview', 'sixthtype-framework' ) .
				'</div>
			  	 <div class="stfr-clear"></div>';

		// font variants (styles)
		echo '<div class="stfr-font-options font-variants">
		    	<h3>' . esc_html( __( 'Font Styles:', 'sixthtype-framework' ) ) . '</h3>
		      </div>';

		// font subsets (character sets)
		echo '<div class="stfr-font-options font-subsets">
		    	<h3>' . esc_html( __( 'Font Character Sets:', 'sixthtype-framework' ) ) . '</h3>
		      </div>';

		$fallback = $option_value['category'];

		if( isset( $option_value['fallback'] ) ) {

			$fallback = $option_value['fallback'];
		}

		// font fallbacks
		echo '<div class="stfr-font-options">
		    	<h3>' . esc_html( __( 'Fallback Fonts:', 'sixthtype-framework' ) ) . '</h3>
		    	<p class="description">' . esc_html( __( 'Enter fallback fonts separated by comma:', 'sixthtype-framework' ) ) . '</p>
			  	<input type="text" class="stfr-input font-fallback" value="' . esc_attr( $fallback ) . '" />
			  </div>';

		echo '</div>';
	}

	/**
	 * Icon Picker
	 *
	 * @access 	public
	 * @since  	1.0
	 * @param  	array $option
	 * @param  	SixthType_Options_View $option_view
	 * @param   mixed $option_value
	 */
	public function icon_picker( $option, $option_view, $option_value ) {

		// display option title
		$option_view->display_option_title( $option );
		
		$icon_font = apply_filters( 'stfr_icon_font', array() );

		if( empty( $icon_font ) ) {

			echo '<p class="stfr-notice">' .
					 __( 'There are no icon fonts available.', 'sixthtype-framework' ) .
				  '</p>';

			return;
		}

		$icons_prefix = 'fa';

		if( isset( $icon_font['prefix'] ) ) {

			$icons_prefix = $icon_font['prefix'];
		}

		$icons_list = $option_view->get_fonts_icons();

		if( is_wp_error( $icons_list ) ) {

			echo '<p class="stfr-notice">' .
					 $icons_list->get_error_message() .
				  '</p>';
			
			return;
		}
			
		echo '<div class="' . esc_attr( $option['class'] ) . '">';
		echo 	'<input type="hidden" name="' . esc_attr( $option['name'] ) . '" value="' . esc_attr( $option_value ) . '"/>';
		
		echo '<div class="stfr-font-icon-preview">
					<span class="' . esc_attr( $icons_prefix . ' ' . $option_value ) . '"></span>
			  </div>';

		echo 	'<div class="stfr-font-icons-list-wrap">
					<ul class="stfr-font-icons-list">';
		
		foreach( $icons_list as $icon ) {

			$selected = '';

			if( $option_value == $icon ) {
				$selected = ' is-active';
			}

			echo '<li class="stfr-font-icon stfr-left' . $selected . '" data-name="' . esc_attr( $icon ) . '">
					<span class="' . esc_attr( $icons_prefix . ' ' . $icon ) . '"></span>
				  </li>';

		}
		
		echo '</ul><div class="stfr-clear"></div>
			  </div>';

		echo '</div>';
	}
}

endif; // end class exists check

?>