<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'SixthType_Plugin_Settings' ) ) :

/**
 * SixthType Plugin Settings
 *
 * @version  1.0
 * @since 	 1.0
 * @package  SixthType-Framework/Includes
 * @author 	 SixthType <hello@sixthtype.com>
 */
class SixthType_Plugin_Settings {

	/**
	 * Save Settings
	 *
	 * @access  public
	 * @since   1.0
	 * @param   array $plugin_settings - plugin options array
	 * @param   array $settings_to_save - options data (commonly from POST request)
	 * @param   string $settings_key - unique plugin settings key
	 * @return  boolean 
	 */
	public function save_settings( $plugin_settings, $settings_to_save, $settings_key ) {

		if( empty( $plugin_settings ) ) {

			return false;
		}

		$plugin_new_settings = array();
		
		foreach( $plugin_settings as $option ) {
			
			if( $option['type'] == 'heading' ){
				
				continue;
			}

			if( array_key_exists( $option['id'], $settings_to_save ) ) {
				
				$value = SixthType_Validator::validate_option( $settings_to_save[ $option['id'] ], $option );
					
				if( is_wp_error( $value ) ) {

					return false;
				}

			} else {

				$value = $option['default'];
					
				if( $option['type'] == 'checkbox' ) {

					$value = 'off';
				}

			}

			$plugin_new_settings[ $option['id'] ] = $value;
		}

		update_option( $settings_key, $plugin_new_settings );
		
		return true;
	}

	/**
	 * Reset Settings
	 *
	 * Set plugin settings defaults
	 *
	 * @access  public
	 * @since   1.0
	 * @param   array $plugin_settings - plugin options array
	 * @param   string $settings_key - unique plugin settings key
	 * @return  boolean
	 */
	public function reset_settings( $plugin_settings, $settings_key ) {

		if( empty( $plugin_settings ) || $settings_key == '' ) {

			return false;
		}

		$settings = array();
		
		foreach( $plugin_settings as $option ) {
		
			if( $option['type'] == 'heading' ){
				continue;
			}

			$settings[ $option['id'] ] = $option['default'];
		}

		update_option( $settings_key, $settings );

		return true;
	}

	/**
	 * Add Setting
	 *
	 * Adds a custom setting value to plugin setting array and
	 * saves it to DB.
	 * 
	 * Use it only if you need to add custom plugin setting
	 * that is not part of the plugin settings options array.
	 *
	 * @access  public
	 * @since   1.0
	 * @param   string $key
	 * @param   mixed  $value
	 * @param   string $settings_key - unique plugin settings key
	 */
	public function add_setting( $key, $value, $settings_key ) {

		if( empty( $key ) || empty( $value ) ||  $settings_key == '' ) {

			return;
		}

		$plugin_settings = get_option( $settings_key );

		if( ! is_array( $plugin_settings ) || array_key_exists( $key, $plugin_settings ) ) {

			return;
		}

		if( is_array( $value ) ) {

			foreach ( $value as $key => $item ) {
				
				$value[ $key ] = wp_kses_post( stripslashes( $item ) );
			}
		
		} else {

			$value = wp_kses_post( stripslashes( $value ) );
		}

		$plugin_settings[ $key ] = $value;

		update_option( $settings_key, $plugin_settings );
	}
}

endif; // end class exists check

?>