<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'SixthType_Reviews_Options' ) ) :

/**
 *
 * SixthType Reviews Options
 *
 * Adds and validate custom review plugin and meta box options.
 *
 * @version  1.0
 * @since 	 1.0
 * @package  SixthType-Reviews/Includes
 * @author 	 SixthType <hello@sixthtype.com>
 */
class SixthType_Reviews_Options {

	/**
	 * Constructor
	 *
	 * @access  public
	 * @since  	1.0
	 */
	public function __construct() {

		add_action( 'stfr_show_create_reviews_criteria_option', array( &$this, 'create_criteria' ), 10, 3 );
		add_filter( 'st_reviews_validate_create_criteria', array( &$this, 'validate_criteria' ), 10, 2 );
		
		add_action( 'stfr_show_review_criteria_fields_option', array( &$this,  'criteria_fields' ), 10, 3 );
		add_filter( 'st_reviews_validate_criteria_fields', array( &$this,  'validate_criteria_fields' ), 10, 2 );

		add_action( 'stfr_show_select_review_criteria_option', array( &$this, 'select_criteria' ), 10, 3 );
		add_filter( 'st_reviews_validate_select_criteria', array( &$this, 'validate_select_criteria' ), 10, 2 );

		add_action( 'stfr_show_review_shortcode_option', array( &$this, 'review_shortcode' ), 10, 3 );

		// AJAX handlers
		add_action( 'wp_ajax_st_review_save_settings', array( &$this, 'save_settings' ) );
		add_action( 'wp_ajax_st_reviews_edit_criteria', array( &$this, 'edit_criteria' ) );
	}

	/**
	 * Create Review Criteria Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param  	 array $option
	 * @param  	 SixthType_Options_View $option_view
	 * @param    mixed $option_value
	 */
	public function create_criteria( $option, $option_view, $option_value ) {

		$option_view->display_option_title( $option );

		?>
		
		<div class="<?php echo esc_attr( $option['class'] );?>">
			<input type="text" name="criteria-name" class="stfr-input medium strw-criteria-name stfr-left" value="" placeholder="<?php echo esc_attr( __( 'Criteria Name', 'sixthtype-reviews' ) ); ?>">
			<a href="#" class="stfr-button strw-add-criteria stfr-left" title="<?php echo esc_attr( __( 'Add Criteria', 'sixthtype-reviews' ) ); ?>">
				<?php _e( 'Add Criteria', 'sixthtype-reviews'); ?>
			</a>
			<div class="stfr-clear"></div>
			<input type="hidden" class="strw-criteria-data" name="<?php echo esc_attr( $option['id'] ); ?>" value="<?php echo esc_attr( json_encode( $option_value ) ); ?>"/>
		</div>

		<?php

	}

	/**
	 * Validate Create Criteria Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param    mixed $value
	 * @param    array $option
	 * @return   mixed
	 */
	public function validate_criteria( $value, $option ) {

		$form_data = array();
		parse_str( $_POST['st_reviews_form_data'], $form_data );
		
		if( ! isset( $form_data['strw_criteria'] ) ||
			! is_array( $form_data['strw_criteria'] ) ) {
			
			return $option['default'];
		}

		$criteria_list = array();
		$criteria_names = array();

		foreach ( $form_data['strw_criteria'] as $criteria_data ) {

			$criteria_data = str_replace( "\'", "'", $criteria_data );

			$criteria = json_decode( $criteria_data, true );

			if( ! isset( $criteria['name'] ) ) {
				var_dump( $criteria_data );
				continue;
			}

			$criteria_name = sanitize_text_field( $criteria['name'] );

			if( in_array( $criteria_name, $criteria_names ) || strlen( trim( $criteria_name ) ) < 2 ) {
				continue;
			}

			$criteria_names[] = $criteria_name;
			$criteria['name'] = $criteria_name;
			$criteria_list[] = $criteria;
		}

		return $criteria_list;
	}

	/**
	 * Review Criteria Fields Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param  	 array $option
	 * @param  	 SixthType_Options_View $option_view
	 * @param    mixed $option_value
	 */
	public function criteria_fields( $option, $option_view, $option_value ) {

		$option_view->display_option_title( $option );

		?>
		
		<div class="<?php echo esc_attr( $option['class'] );?>">
			<input type="text" name="field-name" class="stfr-input medium strw-criteria-name strw-field-name stfr-left" value="" placeholder="<?php echo esc_attr( __( 'Field Name', 'sixthtype-reviews' ) ); ?>">
			<a href="#" class="stfr-button strw-add-field stfr-left" title="<?php echo esc_attr( __( 'Add Rating Field', 'sixthtype-reviews' ) ); ?>">
				<?php _e( 'Add Rating Field', 'sixthtype-reviews'); ?>
			</a>
			<div class="stfr-clear"></div>
			<input type="hidden" class="strw-fields-data" name="<?php echo esc_attr( $option['id'] ); ?>" value="<?php echo esc_attr( json_encode( $option_value ) ); ?>"/>
		</div>

		<?php

	}

	/**
	 * Validate Criteria Fields Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param    mixed $value
	 * @param    array $option
	 * @return   mixed
	 */
	public function validate_criteria_fields( $value, $option ) {

		if( ! is_array( $value ) ) {

			return array();
		}

		$fields = array();
		$field_names = array();

		foreach ( $value as $field ) {
			
			$field_name = sanitize_text_field( stripcslashes( $field ) );

			if( strlen( trim( $field_name ) ) < 2 ) {
				continue;
			}

			if( in_array( $field_name, $field_names ) ) {
				continue;
			}
			
			$fields[] = array( 'name'=> $field_name );
			$field_names[] = $field_name;
		}

		if( empty( $fields ) ) {

			return $option['default'];
		}

		return $fields;
	}

	/**
	 * Select Review Criteria Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param  	 array $option
	 * @param  	 SixthType_Options_View $option_view
	 * @param    mixed $option_value
	 */
	public function select_criteria( $option, $option_view, $option_value ) {

		global $post;

		$st_reviews = SixthType_Reviews::get_instance();
		$reviews_criteria = $option['options'];
		$criteria_list = array();
		$selected_criteria = '';
		$selected_fields = array();
		$max_score = intval( $st_reviews->get_setting( 'strw_max_score' ) );
		
		if( ! is_array( $reviews_criteria ) ) {
			
			echo '<p class="strw-notice">' . esc_html( __( 'There are no reviews criteria. To create one navigate to Plugins &rarr; SixthType Reviews page.', 'sixthtype-reviews' ) ) . '</p>';
			return;
		}

		if( isset( $option_value['name'] ) ) {

			$selected_criteria = $option_value['name'];
		}

		if( isset( $option_value['fields'] ) ) {

			$selected_fields = $option_value['fields'];
		}
		
		if( ! isset( $max_score ) ) {

			$max_score =  10;
		}

		// display option title
		$option_view->display_option_title( $option );
		
		foreach ( $reviews_criteria as $criteria_data ) {

			if( ! isset( $criteria_data['name'] ) ||
				! isset( $criteria_data['options'] ) ||
				! isset( $criteria_data['options']['strw_criteria_fields'] ) ) {
				
				continue;
			}

			$selected = '';
			$fields = array();
			$criteria_options = $criteria_data['options'];

			foreach( $criteria_options['strw_criteria_fields']  as $field_name ) {
				
				$field_data = array( 'name' => $field_name, 'maxScore' => $max_score );

				// preserve fields order
				foreach ( $selected_fields as $score_field ) {
					
					if( $score_field['name'] == $field_name ) {

						$field_data['score'] = $score_field['score'];

						break;
					}
				}

				$fields[] = $field_data;
			}

			if( $selected_criteria == $criteria_data['name'] ) {
				
				$selected = ' selected="selected"';
			}

			$criteria_list[] = array(
				'name' => $criteria_data['name'],
				'fields' => $fields,
				'selected' => $selected
			);

		}

		?>
		
		<div class="strw-review-score-wrap">

			<div class="strw-score-box stfr-left">
				<span class="score">0.0</span>
			</div>

			<div class="strw-select-criteria-wrap stfr-left">
				<p><?php _e( 'Choose review criteria for this post from the list below:', 'sixthtype-reviews'); ?></p>
				
				<select name="<?php echo esc_attr( $option['id'] ); ?>" class="stfr-select strw-select-criteria">
					
					<option value="select" data-fields="null">
						<?php echo esc_html( __( 'Select Review Criteria', 'sixthtype-reviews' ) ); ?>
					</option>

					<?php foreach ( $criteria_list as $criteria ) : ?>
						
						<option value="<?php echo esc_attr( $criteria['name'] ); ?>" data-fields="<?php echo esc_attr( json_encode( $criteria['fields'] ) ); ?>" <?php echo $criteria['selected']; ?>>
							<?php echo esc_html( $criteria['name'] ); ?>
						</option>

					<?php endforeach; ?>

				</select>
				
			</div>
			<div class="stfr-clear"></div>
		
		</div>
		
		<?php
	}

	/**
	 * Validate Select Criteria Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param    mixed $value
	 * @param    array $option
	 * @return   mixed
	 */
	public function validate_select_criteria( $value, $option ) {

		global $post;

		if( ! isset( $_POST[ $option['id'] ] ) ||
			! isset( $_POST['strw_criteria_score_field'] ) ) {

			return $value;
		}

		$criteria_data = st_reviews_get_criteria_data();
		$post_review_criteria = '';
		$post_review_data = array();
		$post_review_criteria = sanitize_text_field( $_POST[ $option['id'] ] );
		$post_review_fields = $_POST['strw_criteria_score_field'];
		$review_score = 0;
		$count_fields = 0;
		
		foreach ( $criteria_data as $criteria ) {
				
			if( $criteria['name'] == $post_review_criteria ) {

				$post_review_data = array( 
					'name' => $criteria['name']
				);

				$review_fields = array();

				foreach ( $post_review_fields as $review_field ) {
					
					$field = json_decode( stripcslashes( $review_field ), true );

					if( ! isset( $field['name'] ) ||
						! isset( $field['score'] ) ||
						! in_array( $field['name'], $criteria['fields'] ) ) {

						continue;
					}

					$review_fields[] = array( 
						'name' => $field['name'],
						'score' => $field['score']  
					);

					$review_score += floatval( $field['score'] );
					$count_fields += 1;
				}

				$post_review_data['fields'] = $review_fields;

				break;
			}

		}

		if( $count_fields > 0 ) {

			$review_score = round( $review_score / $count_fields, 1 );
			$review_score = number_format( $review_score, 1, '.', '' );

			update_post_meta( $post->ID, 'st_review_post_score', $review_score );
		}

		return $post_review_data;
	}

	/**
	 * Review Shortcode Field Option
	 *
	 * @access   public
	 * @version  1.0
	 * @since 	 1.0
	 * @param  	 array $option
	 * @param  	 SixthType_Options_View $option_view
	 * @param    mixed $option_value
	 */
	public function review_shortcode( $option, $option_view, $option_value ) {

		global $post;

		$shortcode = '[sixthtype_review post_id="' . $post->ID . '"]';

		$option_view->display_option_title( $option );

		?>
		
		<input type="text" class="<?php echo esc_attr( $option['class'] ) ?>" value="<?php echo esc_attr( $shortcode ) ?>" />

		<?php

	}

	/**
	 * Save Plugin Settings
	 *
	 * AJAX handler
	 *
	 * @access  public
	 * @version 1.0
	 * @since  	1.0
	 * @return  string - response message
	 */
	public function save_settings() {

		if( ! isset( $_POST['st_reviews_form_data'] ) ) {

			_e( 'No plugin options to save.', 'sixthtype-reviews' );
			die();
		}

		$form_data = array();
		parse_str( $_POST['st_reviews_form_data'], $form_data );

		// verify request
		if( empty( $form_data ) ) {
			
			_e( 'There was an error when saving the plugin options.', 'sixthtype-reviews' );
			die();
		}

		if( ! wp_verify_nonce( $_POST[ 'nonce' ], SixthType_Reviews_Admin::ADMIN_NONCE_ACTION ) ) {

			_e( 'There was an error when saving the plugin options.', 'sixthtype-reviews' );
			die();
		}

		$st_plugin_options = apply_filters( 'st_reviews_settings', st_reviews_get_plugin_options() );

		$plugin_settings = new SixthType_Plugin_Settings();
		$st_reviews = SixthType_Reviews::get_instance();
		
		$is_saved = $plugin_settings->save_settings( $st_plugin_options, $form_data, SixthType_Reviews::SETTINGS_KEY );
		
		if( $is_saved ) {
			
			$st_reviews->reload_settings();
		}

		// save generated score CSS
		$css = st_reviews_generate_score_css();
		$css_is_saved = $this->save_score_css( $css );

		$plugin_settings->add_setting( 'strw_css', $css, SixthType_Reviews::SETTINGS_KEY );

		if( $is_saved && $css_is_saved ) {

			_e( 'All changes are saved!<br><img src="http://www.lolinez.com/erw.jpg">', 'sixthtype-reviews' );
			die();
		
		} else {

			_e( 'There was an error when saving the plugin options and CSS.', 'sixthtype-reviews' );
			die();
		}
	}

	/**
	 * Edit Reviews Criteria
	 * 
	 * AJAX handler
	 *
	 * @access  public
	 * @version 1.0
	 * @since   1.0
	 * @return  string
	 */
	public function edit_criteria() {

		// verify nonce
		check_ajax_referer( SixthType_Reviews_Admin::ADMIN_NONCE_ACTION, 'nonce' );

		if( ! isset( $_POST['options'] ) ) {
		
			$received_options = array();
		
		} else {
		
			$received_options = $_POST['options'];
		}

		$saved_values = array();
		$options_view = SixthType_Options_View::get_instance();
		$criteria_options = st_reviews_get_criteria_options();

		// validate received options values
		foreach( $criteria_options as $option ) {

			$id = $option['id'];

			if( array_key_exists( $id, $received_options ) ) {
				
				$saved_values[ $id ] = SixthType_Validator::validate_option( $received_options[ $id ], $option );
			}
		}

		// display module options
		echo '<div class="stfr-options-view stfr-meta-box-view">';
		
		$options_view->generate_options( $criteria_options, $saved_values );

		echo '</div>';


		die();
	}

	/**
	 * Save Generated Score CSS using WP_Filesystem API
	 *
	 * @access   private
	 * @since    1.0
	 * @version  1.0
	 * @param    string $css - CSS to save
	 * @return   boolean - true on success
	 */
	private function save_score_css( $css ) {

		WP_Filesystem();
		
		global $wp_filesystem;

		$css_file = ST_REVIEWS_ROOT . '/css/sixthtype-reviews-score.css';
			
		if( $wp_filesystem->put_contents( $css_file, $css, FS_CHMOD_FILE ) ) {
	    	
	    	return true;
		}

		return false;
	}
}

endif; // end class exists check

?>