<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'SixthType_Reviews' ) ) :

/**
 * SixthType Reviews
 *
 * Built using Singleton and design pattern principle, which ensures 
 * that at any time you will only have one SixthType_Reviews object 
 * instantiated.
 *
 * Use SixthType_Reviews::get_instance() to obtain singleton object
 * 
 * @version  1.0
 * @since 	 1.0
 * @package  SixthType-Reviews/Includes
 * @author 	 SixthType <hello@sixthtype.com>
 */
class SixthType_Reviews {

	/**
	 * Plugin settings key
	 * 
	 * @access 	public
	 * @since 	1.0
	 * @var 	string
	 */
	const SETTINGS_KEY = 'sixthtype_reviews_plugin_settings';

	/**
	 * Singleton instance
	 *
	 * @static
	 * @access 	private
	 * @since 	1.0
	 * @var 	object - SixthType_Reviews instance
	 */
	private static $instance = null;

	/**
	 * Plugin settings
	 *
	 * @access  private
	 * @since   1.0
	 * @var     array
	 */
	private $settings;

	/**
	 * Holds reviews criteria names
	 * 
	 * @access  private
	 * @since   1.0
	 * @var     array
	 */
	private $reviews_criteria;

	/**
	 * Should plugin load its frontend CSS and JS
	 *
	 * @access  private
	 * @since   1.0
	 * @var     boolean
	 */
	private $enqueue_frontend;

	/**
	 * Constructor
	 *
	 * Use 'st_reviews_enqueue_front_end' filter hook to disable
	 * loading of frontend styles and scripts.
	 * 
	 * @access 	private
	 * @since 	1.0
	 */
	private function __construct() {
		
		$this->settings = array();
		$this->set_reviews_criteria();
		$this->enqueue_frontend = apply_filters( 'st_reviews_enqueue_front_end', true );

		add_action( 'wp_enqueue_scripts', array( &$this, 'load_styles' ) );
		add_filter( 'post_class', array( &$this, 'add_post_css_class' ) );
		add_filter( 'stfr_setup_meta_boxes', array( &$this, 'add_review_meta_box' ) );
		add_filter( 'stfr_setup_posts_meta_boxes', array( &$this, 'posts_meta_boxes' ) );
		add_filter( 'stfr_sh_shortcodes_list', array( &$this, 'add_review_shortcode' ) );
		add_shortcode( 'sixthtype_review', array( &$this, 'review_shortcode' ) );
	}
	
	/**
	 * Get Instance
	 *
	 * Use this method to obtain SixthType_Reviews instance
	 *
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @return 	SixthType_Reviews - instance object
	 */
	public static function get_instance() {

		if ( ! ( self::$instance instanceof SixthType_Reviews ) ) {
      		
      		self::$instance = new SixthType_Reviews();
		}
    	
    	return self::$instance;
	}

	/**
	 * Load Plugin Styles
	 *
	 * Use 'st_review_inline_css_handle' filter hook to set
	 * your own handler to which generated score style will be
	 * appended using wp_add_inline_style().
	 * 
	 * @access public
	 * @since  1.0
	 */
	public function load_styles() {

		if( is_admin() ) {
			
			return;
		}

		$css_handle = apply_filters( 'st_reviews_inline_css_handle', '' );
		
		if( defined( 'ST_THEME_NAME' ) ) {

			$css_handle = 'sttm-css-main';
		}

		if( $css_handle != '' ) {

			$css = $this->get_setting( 'strw_css' );
			
			wp_add_inline_style( $css_handle, $css );

		} else {

			wp_register_style(
				'strw-css-score',
				ST_REVIEWS_URI . '/css/sixthtype-reviews-score.css',
				null,
				'1.0',
				'screen'
			);
    		
    		wp_enqueue_style( 'strw-css-score' );
		}

		if( $this->enqueue_frontend ) {

			wp_register_style(
				'strw-css-main',
				ST_REVIEWS_URI . '/css/sixthtype-reviews.min.css',
				null,
				'1.0',
				'screen'
			);
    		
    		wp_enqueue_style( 'strw-css-main' );
		}

	}

	/**
	 * Add Review Post CSS Class
	 *
	 * Adds 'hreview' microformat class to review posts
	 * 
	 * @access  public
	 * @since   1.0
	 * @param   array $classes
	 * @return  array
	 */
	public function add_post_css_class( $classes ) {

		global $post;

		if( ! $this->is_review_post( $post->ID ) ) {

			return $classes;
		}

		if( is_singular() ) {
			
			array_unshift( $classes, 'hreview' );
		}

		return $classes;
	}

	/**
	 * Add Review Meta Box
	 *
	 * @access  public
	 * @since   1.0
	 * @param   array $meta_boxes
	 * @return  array
	 */
	public function add_review_meta_box( $meta_boxes ) {

		$meta_boxes['st_review'] = array(
			'id' => 'st-plugin-reviews-meta-box',
			'title' => __( 'Review Options', 'sixthtype-reviews' ),
			'context' => 'normal',
			'priority' => 'high',
			'options' => 'st_reviews_get_meta_box_options'
		);

		return $meta_boxes;
	}

	/**
	 * Set Post Types Meta Boxes
	 *
	 * @access  public
	 * @since   1.0
	 * @param   array $post_meta_boxes
	 * @return  array
	 */
	public function posts_meta_boxes( $post_meta_boxes ) {

		$selected_post_types = $this->get_review_post_types();

		if( empty( $selected_post_types ) ) {

			return $post_meta_boxes;
		}

		foreach( $selected_post_types as $post_type ) {
			
			$post_meta_boxes[] = array( 
				'post_type' => $post_type,
				'meta_boxes' => array( 'st_review' )
			);
		}

		return $post_meta_boxes;
	} 

	/**
	 * Add Review Shortcode Inside
	 * SixthType Framework Shortcodes Picker
	 *
	 * @access  public
	 * @since   1.0
	 * @param   array $shortcodes_list
	 * @return  array
	 */
	public function add_review_shortcode( $shortcodes_list ) {
		
		$shortcodes_list['sixthtype_sh_review'] = array(
			'title' => __( 'Review Score', 'sixthtype-reviews' ),
			'icon' => ST_REVIEWS_URI . 'admin/images/review-shortcode.png',
			'name' => 'sixthtype_review',
			'has_content' => false,
			'options' => 'st_reviews_shortcode_options',
			'group_id' => 'review',
			'group_title' => __( 'Review Shortcodes', 'sixthtype-reviews' )
		);

		return $shortcodes_list;
	}

	/**
	 * Set Reviews Criteria
	 *
	 * @access  private
	 * @since   1.0
	 */
	private function set_reviews_criteria() {

		$this->reviews_criteria = array();
		$reviews_criteria = $this->get_criteria();

		foreach( $reviews_criteria as $criteria ) {
			
			if( isset( $criteria['options']['strw_criteria_fields'] ) ) {
				
				$this->reviews_criteria[] = $criteria['name'];
			}
		}
	}

	/**
	 * Get Settings
	 *
	 * Returns an array with plugin settings
	 * 
	 * @access  public
	 * @since   1.0
	 * @return  array - plugin settings
	 */
	public function get_settings() {

		if( empty( $this->settings ) ) {
			
			$settings = get_option( self::SETTINGS_KEY );

			if( ! empty( $settings ) ) {

				$this->settings = $settings;
			}
		}

		return $this->settings;
	}

	/**
	 * Reload Settings
	 *
	 * In case of plugin settings change will reload currently
	 * stored plugin settings in this instance.
	 * This way we will work with correct settings values.
	 *  
	 * @access  public
	 * @since   1.0 
	 */
	public function reload_settings() {

		$this->settings = array();
	}

	/**
	 * Get Plugin Setting
	 *
	 * @access  public
	 * @param   string $id - setting ID
	 * @return  mixed - null if setting does not exists
	 */
	public function get_setting( $id ) {

		$settings = $this->get_settings();

		if( array_key_exists( $id, $this->get_settings() ) ) {
			
			return $settings[ $id ];
		}

		return null;
	}

	/**
	 * Get Review Criteria
	 *
	 * Returns an array with review criteria
	 *
	 * @access  public
	 * @since   1.0
	 * @return  array - saved review criteria
	 */
	public function get_criteria() {

		$criteria = $this->get_setting( 'strw_reviews_criteria' );

		if( ! empty( $criteria ) ) {

			return $criteria;
		}

		return array();
	}

	/**
	 * Get Review Post Types
	 *
	 * @access  public
	 * @since   1.0
	 * @return  array - chosen by user post types
	 */
	public function get_review_post_types() {

		$post_types = $this->get_setting( 'strw_review_post_types' );

		if( isset( $post_types ) ) {

			return $post_types;
		}

		return array();
	}

	/**
	 * Is Review Post
	 *
	 * Determine if post with given ID is review
	 *
	 * @access  public
	 * @since   1.0
	 * @param   integer $post_id
	 * @return  boolean
	 */
	public function is_review_post( $post_id ) {

		global $post;

		if( ! isset( $post_id ) || ! is_numeric( $post_id ) ) {
			
			return false;
		}

		$post_id = intval( $post_id );
		$review_criteria = $this->get_review_criteria( $post_id );

		if( $review_criteria == 'select' ||
			empty( $review_criteria ) ||
			! in_array( $review_criteria['name'] , $this->reviews_criteria ) ) {

			return false;
		}

		$post_type = '';

		if( isset( $post ) && $post->ID == $post_id ) {

			$post_type = $post->post_type;
		
		} else {
		
			$post_data = get_post( $post_id );

			if( isset( $post_data ) ) {

				$post_type = $post_data->post_type;
			}
		}

		if( in_array( $post_type, $this->get_review_post_types() ) ) {

			return true;
		}

		return false;
	}

	/**
	 * Get Review Data
	 *
	 * Will return an array with all review post data.
	 * 
	 * @access  public
	 * @since   1.0
	 * @param   integer $post_id
	 * @return  array
	 */
	public function get_review_data( $post_id ) {

		if( ! $this->is_review_post( $post_id ) ) {

			return array();
		}

		$post = get_post( $post_id );
		$author = get_the_author_meta( 'display_name', $post->post_author ); 
		$date = $post->post_date;
		$max_score = $this->get_setting( 'strw_max_score' );
		$data = $this->get_review_criteria( $post_id );
		$data = $this->validate_rating_fields( $data, $post_id );

		// add CSS classes and percentage for each rating field
		foreach ( $data['fields'] as $key => $rating ) {
				
			$score_class = $this->get_score_css_class( $rating['score'] );
			$score_percent = $this->get_score_percentage( $rating['score'] );
			
			$data['fields'][ $key ]['score_class'] = $score_class;
			$data['fields'][ $key ]['percent'] = $score_percent;
		}

		$score = $this->get_review_score( $post_id );
		$conclusion = $this->get_review_conclusion( $post_id );

		$data['score'] = $score;
		$data['score_class'] = $this->get_score_css_class( $score );
		$data['conclusion'] = $conclusion;
		$data['max_score'] = intval( $max_score );
		$data['author'] = $author;
		$data['date'] = $date;
		
		return $data;
	}

	/**
	 * Get Review Criteria
	 *
	 * Will return an array with criteria name and 
	 * nested array with each field name and score
	 * 
	 * @access  public
	 * @since   1.0
	 * @param   integer $post_id
	 * @return  array
	 */
	public function get_review_criteria( $post_id ) {

		$post_criteria = get_post_meta( $post_id, 'sixthtype_review_selected_criteria', true );
		
		if( ! empty( $post_criteria ) ) {

			return $post_criteria;
		}

		return array();
	}

	/**
	 * Validate Rating Fields
	 *
	 * @access  private
	 * @since   1.0
	 * @param   array $review_data
	 * @param   integer $post_id
	 * @return  array - sorted post fields
	 */
	private function validate_rating_fields( $review_data, $post_id ) {
		
		$criteria_index = array_search( $review_data['name'], $this->reviews_criteria );

		if( $criteria_index === false ) {

			return $review_data;
		}

		$sorted_fields = array();
		$review_score = 0;
		$review_criteria = $this->get_criteria();
		$criteria_fields = $review_criteria[ $criteria_index ]['options']['strw_criteria_fields'];

		// preserve ratings fields order
		foreach( $criteria_fields as $field_name ) {
			
			foreach ( $review_data['fields'] as $rating_field ) {
				
				if( $field_name == $rating_field['name'] ) {

					$sorted_fields[] = $rating_field;
					
					$review_score += floatval( $rating_field['score'] );

					break;
				}
			}
		}

		// find out if criteria rating fields have been deleted
		if( count( $review_data['fields'] ) > count( $sorted_fields ) ) {

			$review_score = round( $review_score / count( $sorted_fields ), 1 );
			$review_score = number_format( $review_score, 1, '.', '' );

			update_post_meta( $post_id, 'st_review_post_score', $review_score );
		}

		$review_data['fields'] = $sorted_fields;
		
		return $review_data;
	}

	/**
	 * Get Review Post Score
	 *
	 * @access  public
	 * @since   1.0
	 * @param   integer $post_id
	 * @return  float - null if post does not have review
	 */
	public function get_review_score( $post_id ) {

		if( ! $this->is_review_post( $post_id ) ) {

			return null;
		}

		$score = get_post_meta( $post_id, 'st_review_post_score', true );

		return $score;
	}

	/**
	 * Get Review Post Conclusion
	 *
	 * @access  public
	 * @since   1.0
	 * @param   integer $post_id
	 * @return  string
	 */
	public function get_review_conclusion( $post_id ) {

		if( ! $this->is_review_post( $post_id ) ) {

			return '';
		}

		$conclusion = get_post_meta( $post_id, 'sixthtype_review_conclusion', true );

		return $conclusion;
	}

	/**
	 * Display Review
	 *
	 * @access  public
	 * @since   1.0
	 * @param   integer $post_id
	 * @param   boolean $return_content
	 * @return  string
	 */
	public function display_review( $post_id, $return_content = false ) {

		if( ! $this->is_review_post( $post_id ) ) {
			
			if( is_user_logged_in() && 
				$return_content &&
				current_user_can( 'publish_posts' ) ) {
				
				$message = sprintf( __( 'Post with ID (%d) have no review to display.', 'sixthtype-reviews' ), $post_id );

				return '<p class="strw-no-review">' . esc_html( $message ) . '</p>';
		}

			return;
		}

		$review_data = $this->get_review_data( $post_id );
		$review_style = $this->get_setting( 'strw_reviews_style' );
		$templates = st_reviews_get_review_templates();
		
		if( ! array_key_exists( $review_style, $templates ) ||
			! isset( $templates[ $review_style ]['display_hook'] ) ) {

			$display_hook = 'st_reviews_tm_classic';
		
		} else {
			
			$display_hook = $templates[ $review_style ]['display_hook'];
		}

		$before = '<div class="strw-review-box strw-clear">';

		$review = apply_filters( 'st_reviews_before_review', $before );
		$review .= apply_filters( $display_hook, $review_data );
		$review .= apply_filters( 'st_reviews_after_review', '</div>' );

		if( ! $return_content ) {
			
			echo $review;
		
		} else {

			return $review; 
		}
	}

	/**
	 * Display Review Shortcode
	 *
	 * @access  public
	 * @since   1.0
	 * @param   array $atts
	 * @return  string
	 */
	public function review_shortcode( $atts ) {

		if( $this->get_setting( 'strw_disable_shortcode' ) == 'on' ) {

			return '';
		}

		$atts = shortcode_atts( array( 'post_id' => 0 ), $atts );

		if( isset( $atts['post_id'] ) && $atts['post_id'] > 0 ) {

			return $this->display_review( $atts['post_id'], true );
		}
	}

	/**
	 * Get Score CSS Class
	 *
	 * @access  public
	 * @since   1.0
	 * @param   float $score
	 * @return  string
	 */
	public function get_score_css_class( $score ) {

		if( ! isset( $score ) || $score <= 0 ) {

			return '';
		}

		// ordered by rating
		$css_classes = array( 
			'strw-low',
			'strw-semi-low',
			'strw-medium',
			'strw-semi-high',
			'strw-high'
		);

		$max_score = $this->get_setting( 'strw_max_score' );
		$ratio = $max_score / count( $css_classes );
		$score_ratio = floatval( $score ) / $ratio;

		$index = ceil( $score_ratio ) - 1;

		if( isset( $css_classes[ $index ] ) ) {

			return $css_classes[ $index ];
		}

		return $css_classes[0];
	}

	/**
	 * Get Score Percentage Depending On Maximum Score
	 * 
	 * @param  float $score
	 * @return integer
	 */
	public function get_score_percentage( $score ) {

		if( $score <= 0 ) {
			
			return 0;
		}

		$max_score = intval( $this->get_setting( 'strw_max_score' ) );

		if( $score >= $max_score ) {

			return 100;
		}

		return $score / ( $max_score / 100 );
	}

	/**
	 * Prevent object clone
	 *
	 * @access 	private
	 * @since  	1.0
	 */
	private function __clone() {}

}

endif; // end class exists check

?>