<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if( ! class_exists( 'SixthType_Validator' ) ) :

/**
 * SixthType Validator
 *
 * Contains methods for input validation
 * 
 * @static
 * @version 	1.0
 * @since 		1.0
 * @package 	SixthType-Framework/Includes
 * @author 		SixthType <hello@sixthtype.com>
 */
class SixthType_Validator {

	/**
	 * Validate Option
	 *
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option - option array
	 * @return 	mixed - validated ot sanitized value upon error WP_Error object
	 */
	public static function validate_option( $value, $option ) {

		if( ! isset( $value ) || empty( $option ) ) {
			
			return new WP_Error(
				'stfr_validate_option',
				__( 'Parameters are required!', 'sixthtype-framework' )
			);
		}

		if( ! isset( $option['type'] ) ) {
			
			return new WP_Error(
				'stfr_validate_option',
				__( 'Wrong option format!', 'sixthtype-framework' )
			);
		}

		switch( $option['type'] ) {

			case "select":
			case "radio":

				return self::validate_select_option( $value, $option );

			break;

			case "checkbox":
			case "switch_toggle": 

				return self::validate_checkbox_option( $value, $option );

			break;

			case "checkboxes":
				
				return self::validate_checkboxes_option( $value, $option );

			break;

			case "select_image_option":

				return self::validate_select_image_option( $value, $option );

			break;

			case "color_picker":
				
				return self::validate_color_picker_option( $value, $option );
				
			break;

			case "file":

				return self::validate_file_option( $value, $option );

			break;

			case "gallery":

				return self::validate_gallery_option( $value, $option );

			break;

			case "slider_input":
				
				return self::validate_slider_input_option( $value, $option );

			break;

			case "font":

				return self::validate_font_option( $value, $option );

			break;

			case "icon_picker":

				return self::validate_icon_picker_option( $value, $option );

			break;

			default:

				if( isset( $option['validate_hook'] ) ) {

					return apply_filters( $option['validate_hook'], $value, $option );

				} else {
					
					return self::validate_text_input( $value, $option );
				}

			break;
		}
	}

	/**
	 * Validate Select Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_select_option( $value, $option ) {

		// make sure that selected value is in options array
		$select_values = array();

		foreach ( $option['options'] as $option_item ) {
			
			$select_values[] = $option_item['value'];
		}

		if( ! in_array( $value, $select_values ) ) {
			
			return new WP_Error( 
				'stfr_validate_option',
				__( 'Incorrect option value.', 'sixthtype-framework' )
			);
		}

		return $value;
	}

	/**
	 * Validate Checkbox Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_checkbox_option( $value, $option ) {

		if( ! in_array( $value, array( 'on', 'off' ) ) ) {
			
			return $option['default'];
		}

		return $value;
	}

	/**
	 * Validate Select Image Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_select_image_option( $value, $option ) {

		// check if value as key is in $option['options'] array
		if( ! array_key_exists( $value , $option['options'] ) ) {
			
			return $option['default'];
		}

		return $value;
	}

	/**
	 * Validate Color Picker Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_color_picker_option( $value, $option ) {

		if( ! preg_match( '/[0-9A-Fa-f]{6}/', $value ) ) {
			
			return $option['default'];
		}

		return $value;
	}

	/**
	 * Validate File Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_file_option( $value, $option ) {

		// validate attachment IDs for absolute integers only
		if( ! is_array( $value ) ) {
			
			$value = json_decode( stripslashes( htmlspecialchars_decode( $value ) ), true );
		}
		
		if( empty( $value ) || ! isset( $value['id'] ) ) {
			
			return $option['default'];
		}

		$value['id'] = absint( $value['id'] );

		return $value;
	}

	/**
	 * Validate Gallery Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_gallery_option( $value, $option ) {

		// validate gallery attachments IDs for abs integers
		
		if( ! is_array( $value ) ) {
			
			$value = $value = json_decode( stripslashes( htmlspecialchars_decode( $value ) ), true );
		}

		if( empty( $value ) ) {
			
			return $option['default'];
		}

		return array_map( 'absint', $value );
	}

	/**
	 * Validate Slider Input Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_slider_input_option( $value, $option ) {

		// check if value is a number
		if( ! is_numeric( $value ) ) {
			
			return $option['default'];
		}

		// check for min/max range
		if( $value < $option['options']['min'] ||
			$value > $option['options']['max'] ) {

			return $option['default'];
		}
			
		return $value;
	}

	/**
	 * Validate Text Input
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	string $value
	 * @param   array $option - optional
	 * @return 	string
	 */
	public static function validate_text_input( $value, $option = array() ) {

		if( ! isset( $value ) || ! is_string( $value ) ) {
			
			return '';
		}

		if( isset( $option['raw'] ) && $option['raw'] == true ) {

			return $value;
		}

		return wp_kses( stripslashes( $value ), SixthType_Utilities::get_kses_allowed_tags() );
	}

	/**
	 * Validate Checkboxes Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_checkboxes_option( $value, $option ) {

		$available_values = array();
		$cb_values = array();

		foreach ( $option['options'] as $option_item ) {
			$available_values[] = $option_item['value'];
		}

		if( ! is_array( $value ) ) {
			return $option['default'];
		}

		foreach( $value as $cb_val ) { 

			if( ! in_array( $cb_val, $available_values ) ) {
				continue;
			}

			$cb_values[] = $cb_val;
		}

		if( empty( $cb_values ) ) {
			return $option['default'];
		}

		return $cb_values;

	}

	/**
	 * Validate Font Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_font_option( $value, $option ) {

		$data = json_decode( stripslashes( htmlspecialchars_decode( $value ) ), true );
		
		if( ! isset( $data['family'] ) ) {
			return $option['default'];
		}
		
		return $data;
	}

	/**
	 * Validate Icon Picker Option
	 * 
	 * @static
	 * @access 	public
	 * @since  	1.0
	 * @param  	mixed $value
	 * @param  	array $option
	 * @return 	mixed
	 */
	public static function validate_icon_picker_option( $value, $option ) {

		return sanitize_html_class( $value, $option['default'] );
	}
}

endif; // end class exists check

?>