<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SixthType Reviews Templates
 *
 * @package  SixthType-Reviews/Templates
 * @author 	 SixthType <hello@sixthtype.com>
 * @version  1.0
 * @since 	 1.0
 */

/**
 * Get Reviews Templates List
 *
 * @version  1.0
 * @since    1.0
 * @return   array
 */
function st_reviews_get_review_templates() {

	$templates = array(
		
		'st-classic' => array(
			'display_hook' => 'st_reviews_tm_classic'
		),

		'st-classic-right' => array(
			'display_hook' => 'st_reviews_tm_classic_right'
		),

		'st-stars' => array(
			'display_hook' => 'st_reviews_tm_stars'
		),

		'st-stars-right' => array(
			'display_hook' => 'st_reviews_tm_stars_right'
		),

		'st-circles' => array(
			'display_hook' => 'st_reviews_tm_circles'
		),

		'st-classic-bold' => array(
			'display_hook' => 'st_reviews_tm_classic_bold'
		),

		'st-classic-clean' => array(
			'display_hook' => 'st_reviews_tm_classic_clean'
		)
	);

	return apply_filters( 'st_reviews_templates_list', $templates );
}

/**
 * Get Review Template Meta Data
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_get_review_meta( $review ) {

	$author = esc_attr( $review['author'] );
	$date = esc_attr( $review['date'] );

	// review author
	$content = '<span class="reviewer">
					<span class="value-title" title="' . $author . '"></span>
				</span>';

	// review date
	$content .= '<span class="dtreviewed">
					<span class="value-title" title="' . $date . '"></span>
				</span>';

	return $content;
}

/**
 * Get Total Score Box
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @param    string $class - optional CSS class for score box
 * @return   string
 */
function st_reviews_tm_get_total_score( $review, $class = '' ) {

	$content = st_reviews_tm_get_review_meta( $review );

	$score = esc_html( $review['score'] );
	$score_class =  esc_attr( $class . ' ' . $review['score_class'] );
	$score_label = esc_html( __( 'score', 'sixthtype-reviews' ) );
	$max_score = esc_attr( $review['max_score'] );

	// total score
	$content .= '<div class="strw-score-box ' . $score_class . '">
					<span class="strw__total-score rating">
						<span class="value">' . $score . '</span>
						<span class="worst">
							<span class="value-title" title="0.5"></span>
						</span>
						<span class="best">
							<span class="value-title" title="' . $max_score . '"></span>
						</span>
					</span>
					<span>' . $score_label . '</span>
				</div>';

	return $content;
}

/**
 * Genrate Review Icon Rating
 *
 * @version  1.0
 * @since    1.0
 * @param    float $score 
 * @param    array $review 
 * @param    array $icon_classes
 * @return   string      
 */
function st_reviews_tm_get_icon_rating( $score, $review, $icon_classes ) {

	$content = '';
	$full_icons = floor( $score );
	$half_icon = $score - $full_icons;
	$empty_icons = intval( $review['max_score'] - round( $score ) );
	$i = 0;

	// display full stars
	for( $i = 0; $i < $full_icons; $i++ ) {
		
		$content .= '<span class="strw-icon ' . $icon_classes['full'] . '"></span>';
	}

	// display half stars
	if( $half_icon > 0 ) {
		
		$content .= '<span class="strw-icon ' . $icon_classes['half'] . '"></span>';
	}

	// display empty stars
	for( $i = 0; $i < $empty_icons; $i++ ) {
		
		$content .= '<span class="strw-icon ' . $icon_classes['empty'] . '"></span>';
	}

	return $content;
}

/**
 * Classic Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_classic( $review ) {

	// get total score box
	$content = st_reviews_tm_get_total_score( $review );

	$content .= '<ul class="strw-rating-list">';

	// rating fields
	foreach ( $review['fields'] as $field ) {
		
		$name = esc_html( $field['name'] );
		$score = esc_html( $field['score'] );
		$class = esc_attr( $field['score_class'] );
		$percent = 'style="width:' . esc_attr( $field['percent'] ).'%;"';

		$content .= '<li class="strw-rating">
						<span class="strw-rating__label">' . $name . ': 
							<span class="strw-rating__score">' . $score . '</span>
						</span>';
		
		$content .= 	'<div class="strw-bar">
							<div class="strw-score-bar ' . $class .'" ' . $percent . '></div>
						</div>';
		
		$content .= '</li>';
	}

	$content .= '</ul>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {
	
		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
	}

	return $content;
}
add_filter( 'st_reviews_tm_classic', 'st_reviews_tm_classic' );

/**
 * Classic Floated Right Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_classic_right( $review ) {

	// get total score box
	$content = st_reviews_tm_get_total_score( $review, 'strw-left' );

	$content .= '<ul class="strw-rating-list strw-rating-list--left">';

	// rating fields
	foreach ( $review['fields'] as $field ) {
		
		$name = esc_html( $field['name'] );
		$score = esc_html( $field['score'] );
		$class = esc_attr( $field['score_class'] );
		$percent = 'style="width:' . esc_attr( $field['percent'] ).'%;"';

		$content .= '<li class="strw-rating">
						<span class="strw-rating__label">' . $name . ': 
							<span class="strw-rating__score">' . $score . '</span>
						</span>';
		
		$content .= 	'<div class="strw-bar">
							<div class="strw-score-bar ' . $class .'" ' . $percent . '></div>
						</div>';
		
		$content .= '</li>';
	}

	$content .= '</ul>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {
	
		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
	}

	return $content;

}
add_filter( 'st_reviews_tm_classic_right', 'st_reviews_tm_classic_right' );

/**
 * Stars Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_stars( $review ) {

	$icon_classes = array(
		'full' => 'strw-star',
		'half' => 'strw-star--half',
		'empty' => 'strw-star--empty'
	);

	// get total score box
	$content = st_reviews_tm_get_total_score( $review );

	$content .= '<ul class="strw-rating-list">';

	// rating fields
	foreach ( $review['fields'] as $field ) {
		
		$name = esc_html( $field['name'] );
		$score = esc_html( $field['score'] );

		$content .= '<li class="strw-rating">
						<span class="strw-rating__label">' . $name . ': 
							<span class="strw-rating__score">' . $score . '</span>
						</span>';
		
		// generate rating icons
		$content .= st_reviews_tm_get_icon_rating( $score, $review, $icon_classes );

		$content .= '</li>';
	}

	$content .= '</ul>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {
	
		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
	}

	return $content;

}
add_filter( 'st_reviews_tm_stars', 'st_reviews_tm_stars' );

/**
 * Stars Right Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_stars_right( $review ) {
	
	$icon_classes = array(
		'full' => 'strw-star',
		'half' => 'strw-star--half',
		'empty' => 'strw-star--empty'
	);

	// get total score box
	$content = st_reviews_tm_get_total_score( $review, 'strw-left' );

	$content .= '<ul class="strw-rating-list strw-rating-list--left">';

	// rating fields
	foreach ( $review['fields'] as $field ) {
		
		$name = esc_html( $field['name'] );
		$score = esc_html( $field['score'] );
		
		$content .= '<li class="strw-rating">
						<span class="strw-rating__label">' . $name . ': 
							<span class="strw-rating__score">' . $score . '</span>
						</span>';
		
		// generate rating icons
		$content .= st_reviews_tm_get_icon_rating( $score, $review, $icon_classes );

		$content .= '</li>';
	}

	$content .= '</ul>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {
	
		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
	}

	return $content;

}
add_filter( 'st_reviews_tm_stars_right', 'st_reviews_tm_stars_right' );

/**
 * Circles Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_circles( $review ) {

	$icon_classes = array(
		'full' => 'strw-circle',
		'half' => 'strw-circle--half',
		'empty' => 'strw-circle--empty'
	);

	// get total score box
	$content = st_reviews_tm_get_total_score( $review, 'strw-score-box--circle' );
	
	$content .= '<ul class="strw-rating-list">';

	// rating fields
	foreach ( $review['fields'] as $field ) {
		
		$name = esc_html( $field['name'] );
		$score = esc_html( $field['score'] );
		
		$content .= '<li class="strw-rating">
						<span class="strw-rating__label">' . $name . ':
							<span class="strw-rating__score">' . $score . '</span>
						</span>';
		
		// generate rating icons
		$content .= st_reviews_tm_get_icon_rating( $score, $review, $icon_classes );

		$content .= '</li>';
	}

	$content .= '</ul>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {
	
		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
	}

	return $content;

}
add_filter( 'st_reviews_tm_circles', 'st_reviews_tm_circles' );

/**
 * Classic Bold Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_classic_bold( $review ) {

	// get total score box
	$content = st_reviews_tm_get_total_score( $review );

	$content .= '<ul class="strw-rating-list strw-rating--bold">';

	// rating fields
	foreach ( $review['fields'] as $field ) {
		
		$name = esc_html( $field['name'] );
		$score = esc_html( $field['score'] );
		$class = esc_attr( $field['score_class'] );
		$percent = 'style="width:' . esc_attr( $field['percent'] ).'%;"';

		$content .= '<li class="strw-rating">';
		
		$content .= 	'<div class="strw-bar">
							<div class="strw-score-bar ' . $class .'" ' . $percent . '>
								<span class="strw-rating__label">' . $name . ': 
									<span class="strw-rating__score">' . $score . '</span>
								</span>
							</div>
						</div>';
		
		$content .= '</li>';
	}

	$content .= '</ul>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {
	
		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
	}

	return $content;

}
add_filter( 'st_reviews_tm_classic_bold', 'st_reviews_tm_classic_bold' );

/**
 * Classic Clean Review Template
 *
 * @version  1.0
 * @since    1.0
 * @param    array $review
 * @return   string
 */
function st_reviews_tm_classic_clean( $review ) {

	$content = st_reviews_tm_get_review_meta( $review );

	$score = esc_html( $review['score'] );
	$score_class =  esc_attr( $review['score_class'] );
	$max_score = esc_attr( $review['max_score'] );

	// total score
	$content .= '<div class="strw-score-box strw-score-box--clean ' . $score_class . '">
					<span class="strw__total-score rating">
						<span class="value">' . $score . ' / ' . $max_score . '</span>
						<span class="worst">
							<span class="value-title" title="0.5"></span>
						</span>
						<span class="best">
							<span class="value-title" title="' . $max_score . '"></span>
						</span>
					</span>
				</div>';

	// conclusion
	if( ! empty( $review['conclusion'] ) ) {

		$content .= '<div class="strw-conclusion">' . $review['conclusion'] . '</div>';
}

	return $content;
}
add_filter( 'st_reviews_tm_classic_clean', 'st_reviews_tm_classic_clean' );

?>