<?php

// Exit if accessed directly
if ( !defined( 'ABSPATH' ) ) exit;

/**
 * Model Class
 *
 * Handles generic plugin functionality, mostly Social Network related
 * for all the different features for the autoposting.
 *
 * @package Social Auto Poster
 * @since 1.0.0
 */
class Wpw_Auto_Poster_Model {

	public function __construct() {
	
	}
	
	/**
	 * similar to checked() but checks for array
	 *	
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */	
	function checked_array( $checked, $current ) {

		if ( is_array( $current ) ) {
            if( in_array( $checked, $current ) ) {
                echo ' checked="checked"';
            }
        } else {
            if( $checked == $current ) {
                echo ' checked="checked"';
            }
        }
	}
	
	/**
	 * Get Unserialize the data
	 * 
	 * Handle serialize data and return unserialize data
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	function get_unserialize_data( $data ) {
	
		$undata = unserialize( $data );
		return $undata;		
	}
	
	/**
	 * Escape Tags & Slashes with URL
	 *
	 * Handles escapping the slashes and tags
	 *
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */   
	function escape_url( $data ) {
	
		return esc_url( $data ); // esc_url will do stripslashes and esc_attr both so we dont need to call it.
	}
	
	/**
	 * Escape Tags & Slashes
	 *
	 * Handles escapping the slashes and tags
	 *
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */   
	public function wpw_auto_poster_escape_attr($data){
		return esc_attr_e(stripslashes($data));
	}
	
	/**
	 * Stripslashes
	 * 
	 * It will stripslashes from the content
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */	
	public function wpw_auto_poster_stripslashes_deep( $data = array(), $flag=false ) {
		
		if($flag != true) {
			$data = $this->wpw_auto_poster_nohtml_kses($data);
		}
		$data = stripslashes_deep($data);
		return $data;
	}
	
	/**
	 * Strip Html Tags 
	 * 
	 * It will sanitize text input (strip html tags, and escape characters)
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_nohtml_kses( $data = array() ) {	
		
		if ( is_array($data) ) {
			
			$data = array_map(array($this,'wpw_auto_poster_nohtml_kses'), $data);
			
		} elseif ( is_string( $data ) ) {
			
			$data = wp_filter_nohtml_kses($data);
		}
		
		return $data;
	}	
	
	/**
	 * HTML Entity Decode
	 * 
	 * Handles to decode HTML entities
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_html_decode( $string ) {
		
		return html_entity_decode( $string );
	}
	/**
	 * Get Shortner Link
	 * 
	 * Handles to return shortner link
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	function wpw_auto_poster_get_short_post_link( $link = '', $unique = 'false', $postid = '', $customlink = 'false' ) { //
		
		global $wpw_auto_poster_options;
		
		$shortnertype = $wpw_auto_poster_options['url_shortener'];
		
		$wpw_auto_poster_bitly_username = $wpw_auto_poster_options['bitly_username'];
		$wpw_auto_poster_bitly_api_key = $wpw_auto_poster_options['bitly_api_key'];
		
		//default link when it is blank
		$link = !empty( $link ) ? $link : get_permalink( $postid );
		
		if( $unique == 'true' && $customlink == 'true' ) {//unique url && it is custom url
			$link = add_query_arg( 'wpwautoposter', time(), $link );
		}
		
		switch( $shortnertype ) {
			case 'tinyurl':
				require_once( WPW_AUTO_POSTER_DIR . '/includes/shorteners/tinyurl.php' );
				
				$tinyurl = new wpw_auto_poster_tw_tinyurl;
				$link = $tinyurl->shorten( $link );
				break;
			case 'wordpress':
				
				if( $customlink != 'true' ) { //check custom link should not set
					
					if( get_option('permalink_structure') != '' ) {
						$link = wp_get_shortlink( $postid );
					} else {
						$link = get_permalink( $postid );
					}
					if( $unique == 'true') { //unique url
						$link = add_query_arg( 'wpwautoposter', time(), $link );
					}
					
				}
				break;
			case 'bitly':		
				require_once( WPW_AUTO_POSTER_DIR . '/includes/shorteners/bitly.php' );
				$bitlyurl = new wpw_auto_poster_tw_bitly( $wpw_auto_poster_bitly_username, $wpw_auto_poster_bitly_api_key );
				$link = $bitlyurl->shorten( $link );
				break;
		}
		return $link;
	}
	/**
	 * Replace Shortcode In Twitter Template
	 * 
	 * Handles to return template with replace its shortcodes
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_tweet_status( $post, $template, $title = '' ) {
			
		global $wpw_auto_poster_options;
		
		$prefix = WPW_AUTO_POSTER_META_PREFIX;	
		
		$ispublished = get_post_meta( $post->ID, $prefix . 'tw_status', true );
		$unique = 'false';
		
		if( isset( $ispublished ) && $ispublished == '1' ) { //if post is published on facebook once then change url to prevent duplication
			//unique link for posting
			$unique = 'true';
		}
		
		$postlink = get_permalink( $post->ID );	
		$postlink = $this->wpw_auto_poster_get_short_post_link( $postlink, $unique, $post->ID, 'false' );
		
		$posttitle = $post->post_title;
		$userdata = get_userdata($post->post_author);
		$nicename = $userdata->user_nicename;
		$first_name = get_user_meta( $post->post_author, 'first_name', true);
		$last_name = get_user_meta( $post->post_author, 'last_name', true);
		$fullauthor = $first_name . ' ' . $last_name;
		$posttype = $post->post_type;
		
		//if title is passed from function parameter then use that title
		$posttitle = !empty($title) ? $title : $posttitle;
		
		$replacetags = array( '[link]','[title]', '[full_author]', '[nickname_author]', '[post_type]');
		$replaceval = array( $postlink, $posttitle, $fullauthor, $nicename, $posttype );
		$status = str_replace($replacetags, $replaceval, $template);
		
		return $status;
	}
	/**
	 * Return Template Text from Value
	 * 
	 * Handles to return Template
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_get_tweet_template($tweet_template) {
	
		global $wpw_auto_poster_options;
		$retval = '';
		
		switch ( $tweet_template ) {
			
			case 'title_fullauthor_link'	: 
					$retval = '[title] by [full_author] - [link]';
					break;
			
			case 'title_nickname_link'		:
					$retval = '[title] by @[nickname_author] - [link]';
					break;
					
			case 'post_type_title_link'		:
					$retval = 'New [post_type]: [title] - [link]';
					break;
			
			case 'post_type_title_fullauthor_link'	:
					$retval = 'New [post_type]: [title] by [full_author] - [link]';
					break;
					
			case 'post_type_title_nickname_link'	:
					$retval = 'New [post_type]: [title] by [nickname_author] - [link]';
					break;
			case 'custom'							:
					$retval = $wpw_auto_poster_options['tw_custom_tweet_template'];
					break;
			case  'title_link' 						:
			default :
					$retval = '[title] - [link]';
					break;
		}
		return $retval;
	
	}
	
	/**
	 * Get Self URL
	 * 
	 * Handles to return current URL
	 * 
	 * @package Social Auto Poster
	 * @since 1.0.0
	 */
	public function wpw_auto_poster_self_url() 
	{ 
	    $s = empty($_SERVER["HTTPS"]) ? '' : ($_SERVER["HTTPS"] == "on") ? "s" : ""; 
	    $str1 = strtolower($_SERVER["SERVER_PROTOCOL"]);
	    $str2 = "/";
	    $protocol = substr($str1, 0, strpos($str1, $str2)).$s;
	    $port = ($_SERVER["SERVER_PORT"] == "80") ? "" : (":".$_SERVER["SERVER_PORT"]); 
	    return $protocol."://".$_SERVER['SERVER_NAME'].$port.$_SERVER['REQUEST_URI']; 
	} 
	
}