<?php
/**
 * LaraClassified - Geo Classified Ads CMS
 * Copyright (c) Mayeul Akpovi. All Rights Reserved
 *
 * Email: mayeul.a@larapen.com
 * Website: http://larapen.com
 *
 * LICENSE
 * -------
 * This software is furnished under a license and may be used and copied
 * only in accordance with the terms of such license and with the inclusion
 * of the above copyright notice. If you Purchased from Codecanyon,
 * Please read the full License from here - http://codecanyon.net/licenses/standard
 */

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Input;
use ChrisKonnertz\OpenGraph\OpenGraph;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\View;
use Jaybizzle\CrawlerDetect\CrawlerDetect;
use Torann\LaravelMetaTags\Facades\MetaTag;
use Larapen\CountryLocalization\Helpers\Country;
use Crypt;

class FrontController extends Controller
{
    public $data = array();
    
    protected $cache_expire = 3600; // 1h
    
    protected $country;
    protected $cookie_expire;
    
    public function __construct(Request $request)
    {
        // Cache duration setting
        $this->cache_expire = (int)config('settings.app_cache_expire');
        
        
        // Get User, Country, Domain, lang, ... infos from GeoLocalization middleware /========
        $this->user = $request->get('mw_user');
        $this->country = $request->get('mw_country');
        $this->lang = $request->get('mw_lang');
        
        // Default language for Bots
        $crawler = new CrawlerDetect();
        if ($crawler->isCrawler()) {
            $this->lang = $this->country->get('lang');
            View::share('lang', $this->lang);
            App::setLocale($this->lang->get('abbr'));
        }
        
        // Set Local
        if (!$this->lang->isEmpty()) {
            setlocale(LC_ALL, $this->lang->get('locale'));
        }
        
        // Set Language for Countries /========================================================
        $this->country = Country::trans($this->country, $this->lang->get('abbr'));
        View::share('country', $this->country);
        
        // DNS Prefetch meta tags
        $dns_prefetch = [
            '//fonts.googleapis.com',
            '//graph.facebook.com',
            '//google.com',
            '//apis.google.com',
            '//ajax.googleapis.com',
            '//www.google-analytics.com',
            '//pagead2.googlesyndication.com',
            '//gstatic.com',
            '//cdn.api.twitter.com',
            '//oss.maxcdn.com',
        ];
        View::share('dns_prefetch', $dns_prefetch);
        
        
        // SEO /===============================================================================
        if (config('settings.app_slogan')) {
            $title = config('settings.app_slogan');
        } else {
            $title = t('Free local classified ads in :location', ['location' => $this->country->get('name')]);
        }
        $description = str_limit(str_strip(t('Sell and Buy products and services on :app_name in Minutes',
                ['app_name' => mb_ucfirst(config('settings.app_name'))]) . ' ' . $this->country->get('name') . '. ' . t('Free ads in :location',
                ['location' => $this->country->get('name')]) . '. ' . t('Looking for a product or service') . ' - ' . $this->country->get('name')),
            200);
        
        // Meta Tags
        MetaTag::set('title', $title);
        MetaTag::set('description', $description);
        
        
        // Open Graph /=========================================================================
        //->localeAlternate(['en_US']) // @todo: Get all Language locale
        $this->og = new OpenGraph();
        $this->og->siteName(config('settings.app_name'))->locale($this->lang->has('locale') ? $this->lang->get('locale') : 'en_US')->type('website')->url(url()->current());
        View::share('og', $this->og);
        
        
        // CSRF Control /========================================================================
        // CSRF - Some JavaScript frameworks, like Angular, do this automatically for you.
        // It is unlikely that you will need to use this value manually.
        $domain = (getSubDomainName() != 'www') ? strtolower($this->country->get('code')) . '.' . getDomain() : getDomain();
        setcookie('X-XSRF-TOKEN', csrf_token(), $this->cookie_expire, '/', $domain);
        
        
        // Theme selection /=====================================================================
        if (Input::has('theme')) {
            if (file_exists(public_path() . '/assets/css/style/' . Input::get('theme') . '.css')) {
                config(['app.theme' => Input::get('theme')]);
            } else {
                config(['app.theme' => config('settings.app_theme')]);
            }
        } else {
            config(['app.theme' => config('settings.app_theme')]);
        }
        
        
        // Set config /==========================================================================
        // Set Mail config
        config(['mail.from.address' => config('settings.app_email')]);
        config(['mail.from.name' => config('settings.app_name')]);
    }
}
